/* File: Yn.hpp
 *
 * This class defines the collocation conditions
 *
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-11-15
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef YN_HPP
#define YN_HPP

#include "LSCMConfig.hpp"
#include "Grid.hpp"
#include "QuadratureRule.hpp"
#include <Eigen/Dense>
#include <vector>
#include <memory>
#include <iostream>
#include <cstdlib>

namespace LSCM {

/**
 * This class defines the collocation conditions
 * 
 * We assume that the order of approximation (and therefore the local
 * collocation conditions) can depend on the subinterval of the grid.
 */
class Yn {
public:
    /**
      * This type contains the definitions of the norms in image space, that is,
      * the collocation criterion used.
      */
    typedef enum { LSQ_RN,  /**< pointwise collocation in \f$R^m\f$ */
                   LSQ_INT, /**< use numerical integration */
                   LSQ_L2   /**< use polynomial interpolation */
                  } LSQnorm;

private:
    // local approximation spaces on each subinterval
    std::shared_ptr<std::vector<double>> vtau = nullptr;
    
    // collocation criterion
    LSQnorm lsqnorm;
    // quadrature nodes
    std::shared_ptr<QuadratureRule> intmeth = nullptr;
    // normdata: dependent on functional
    std::shared_ptr<std::vector<double>> iweights = nullptr; // LSQ_INT
    std::shared_ptr<Eigen::MatrixXd> mass = nullptr;         // LSQ_L2
        
public:
    /**
     * Constructor
     * 
     * In each subinterval, the nodes provided by an integration rule
     * are chosen
     * 
     * @param[in] intmeth special collocation points to be used
     * @param[in] lsqnorm collocation criterion
     */
    Yn(std::shared_ptr<QuadratureRule> intmeth, Yn::LSQnorm lsqnorm = LSQ_RN);
        
    /**
     * Constructor
     * 
     * In each subinterval, M Gauss nodes are chosen
     * 
     * @param[in] M the number of collocation points to be used on every
     *               subinterval
     * @param[in] lsqnorm collocation criterion
     */
    Yn(LSCMint M, Yn::LSQnorm lsqnorm = LSQ_RN);
        
    /**
     * Return numer of collocation points
     * 
     */
    LSCMint getM() const { return (*vtau).size(); }
    
    /**
     * Returns set of collocation points
     * 
     */
    std::shared_ptr<std::vector<double>> gettau() { return vtau; }
    
    /**
     * Returns integration weights corresponding to the collocation points
     * 
     */
    std::shared_ptr<std::vector<double>> getinteg(){
        if (lsqnorm != LSQ_INT) {
            std::cerr << "Yn: Wrong functional!" << std::endl;
            std::exit(1);
        }
        return iweights;
    }
    
    /**
     * Returns mass matrix corresponding to the collocation points
     * 
     */
    std::shared_ptr<Eigen::MatrixXd> getmass() {
        if (lsqnorm != LSQ_L2) {
            std::cerr << "Yn: Wrong functional!" << std::endl;
            std::exit(1);
        }
        return mass;
    }
    
    /**
     * Returns collocation criterion
     */
    LSQnorm getnorm() const { return lsqnorm; }
};

}

#endif
