/* File: Yn.hpp
 *
 * This class defines the collocation conditions
 *
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-11-15
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Yn.hpp"
#include "QuadratureRule.hpp"
#include "GaussLegendre.hpp"
#include <vector>
#include <memory>
#include <iostream>
#include <cstdlib>

using namespace std;
using namespace Eigen;

LSCM::Yn::Yn(shared_ptr<QuadratureRule> imm, Yn::LSQnorm lsqnorm_) : lsqnorm(lsqnorm_), intmeth(imm) {
    if (intmeth == nullptr) {
        cerr << "Error Yn: Quadrature rule must be given explicitely" << endl;
        exit(1);
    }
    LSCMint M = (intmeth->getnodes()).size();
    QuadratureRule::IntMethod im = intmeth->getMethod();
    vtau = make_shared<vector<double>>(M);
    for (LSCMint i = 0; i < M; ++i) (*vtau)[i] = im[i].zero;
    
    switch (lsqnorm) {
        case LSQ_RN:
            return;
        case LSQ_INT:
        {
            iweights = make_shared<vector<double>>(M);
            for (LSCMint i = 0; i < M; ++i) (*iweights)[i] = im[i].weight;
            break;
        }
        case LSQ_L2:
        {
            mass = make_shared<MatrixXd>(intmeth->gen2mass());
            break;
        }
        default:
            // Cannot happen
            cerr << "Yn: Unknown functional" << endl;
            exit(1);
    }
}

LSCM::Yn::Yn(LSCMint M, Yn::LSQnorm lsqnorm_) : lsqnorm(lsqnorm_) {
    if (M <= 0) {
        cerr << "Error Yn: M must be positive" << endl;
        exit(1);
    }
    intmeth = make_shared<GaussLegendre>(M);
    QuadratureRule::IntMethod im = intmeth->getMethod();
    vtau = make_shared<vector<double>>(M);
    for (LSCMint i = 0; i < M; ++i) (*vtau)[i] = im[i].zero;
    
    switch (lsqnorm) {
        case LSQ_RN:
            return;
        case LSQ_INT:
        {
            iweights = make_shared<vector<double>>(M);
            for (LSCMint i = 0; i < M; ++i) (*iweights)[i] = im[i].weight;
            break;
        }
        case LSQ_L2:
        {
            mass = make_shared<MatrixXd>(intmeth->gen2mass());
            break;
        }
        default:
            // Cannot happen
            cerr << "Yn: Unknown functional" << endl;
            exit(1);
    }
}
