/* File: Xn.hpp
 *
 * This class defines the approximation space
 *
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-11-08
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef XN_HPP
#define XN_HPP

#include "LSCMConfig.hpp"
#include "Grid.hpp"
#include "QuadratureRule.hpp"
#include "Basis.hpp"
#include <Eigen/Dense>
#include <vector>
#include <memory>

namespace LSCM {

/**
 * This class defines the approximation space
 * 
 * We assume that the order of approximation (and therefore the local
 * approximation space) can depend on the subinterval of the grid.
 */
class Xn {
private:
    std::shared_ptr<Grid> grid;
    // local approximation spaces on each subinterval
    std::shared_ptr<Basis> vBasis;
    //
    std::vector<bool> D;
    // These are the standard variables
    LSCMint k = 0, m, N;
    
public:
    /**
     * Constructor
     * 
     * @param[in] D vecor indicating differential components
     * @param[in] grid a valid grid 
     * @param[in] basis the local space to be used on every subinterval
     */
    Xn(std::vector<bool> D, std::shared_ptr<Grid> grid,
       std::shared_ptr<Basis> basis);
   
    /**
     * Queury for D
     * 
     * @returns D
     */
    std::vector<bool>& getDptr() { return D; }
    
    /**
     * Queury for grid
     * 
     * @returns grid
     */
    std::shared_ptr<Grid> getgrid() { return grid; }
    
    std::shared_ptr<Basis> getBasis() { return vBasis; }
    
    /**
     * Computation of the whole vector of polynomial values
     * 
     * @param[in] t poLSCMint in the interval
     * @returns vector of polynomial values
     */
    Eigen::VectorXd eval(double t) const;
    
     /**
     * Computation of the whole vector of polynomial values in a given subinterval
     * scaled by its length
     * 
     * @param[in] tau poLSCMint in the subinterval (0 <= tau <= 1)
     * @param[in] h length of subinterval j
     * @returns vector of polynomial values
     */
    Eigen::VectorXd evalsub(double tau, double h) const;
    
   /**
     * Computation of the whole vector of polynomial values at
     * t[j]+0
     * 
     * @param[in] j grid poLSCMint in the interval (0 <= j < n-1)
     * @returns vector of polynomial values
     */
    Eigen::VectorXd evalr(LSCMint j) const;
    
    /**
     * Computation of the whole vector of polynomial values at
     * t[j]-0
     * 
     * @param[in] j grid poLSCMint in the interval (0 < j <= n)
     * @returns vector of polynomial values
     */
    Eigen::VectorXd evall(LSCMint j) const;
    
    /**
     * Computation of the whole vector of derivative values at t
     * 
     * @param[in] t poLSCMint in the interval
     * @returns vector of derivative values
     */
    Eigen::VectorXd deval(double t) const;
    
     /**
     * Computation of the derivatives
     * 
     * @param[in] t poLSCMint in the subinterval (0 <= tau <= 1)
     * @param[in] h length of subinterval j
     * @returns vector of derivative values
     */
    Eigen::VectorXd devalsub(double t, double h) const;
    
   /**
     * The local dimension
     * 
     * @returns the number of unknowns in that subinterval
     */
    LSCMint getdim() const { return (vBasis->getdim())*m+k; }
    
    /**
     * Local polynomial order
     * 
     * @returns local dimension N
     */
    LSCMint getN() const { return N; }
    
    /**
     * dimension of the DAE
     * 
     * @returns the dimension of the DAE
     */
    LSCMint getm() const { return m; }
    
    /**
     * Number of differential variables
     * 
     * @returns the number of differential variables
     */
    LSCMint getk() const { return k; }
    
    /**
     * Find local integration rule
     * 
     * @returns local integration rule
     */
    const QuadratureRule::IntMethod& getsqrtw() {
        return vBasis->getsqrtw();
    }

};

}

#endif
