/* File: Xn.cpp
 *
 * This class defines the approximation space
 *
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-11-08
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Xn.hpp"
#include <Eigen/Dense>
#include <vector>
#include <memory>
#include <iostream>
#include <cstdlib>

using namespace std;
using namespace Eigen;

LSCM::Xn::Xn(std::vector<bool> D_, shared_ptr<Grid> grid_, shared_ptr<Basis> basis_) :
D(D_), grid(grid_), vBasis(basis_) {
    LSCMint n = grid->getn();
    N = vBasis->getdim();
    m = D.size();
    for (LSCMint i = 0; i < m; i++)
        if (D[i]) k++;
}

VectorXd LSCM::Xn::eval(double t) const {
    LSCMint i = grid->interv(t)-1;
    double h = (*grid)[i+1]-(*grid)[i];
    double tau = (t-(*grid)[i])/h;
    
    return evalsub(tau,h);
}

VectorXd LSCM::Xn::evalsub(double tau, double h) const {
    // Should we check for 0 <= tau <= 1?
    
    // Compute function values
    VectorXd algtau = vBasis->evalPa(tau);
    VectorXd difftau = vBasis->evalPd(tau);
    
    // Copy to final result
    VectorXd res(getdim());
    LSCMint idx = 0;
    for (LSCMint j = 0; j < m; ++j) {
        // Scaling from [0,1] to h appears here!
        if (D[j]) {
            res.segment(idx,N+1) = h*difftau;
            idx += N+1;
        }
        else {
            res.segment(idx,N) = algtau;
            idx += N;
        }
    }
    
    // Done
    return res;
}

VectorXd LSCM::Xn::evalr(LSCMint i) const {
    double h = (*grid)[i+1]-(*grid)[i];
    
    // Compute function values
    VectorXd algtau = vBasis->evalPa(0.0);
    //VectorXd difftau = vBasis[i]->evalPd(0.0); // It's the unit vector!
    
    // Copy to final result
    VectorXd res(getdim());
    LSCMint idx = 0;
    for (LSCMint j = 0; j < m; ++j) {
        // Scaling from [0,1] to h appears here!
        if (D[j]) {
            res(idx) = h;
            res.segment(idx+1,N) = VectorXd::Zero(N);
            idx += N+1;
        }
        else {
            res.segment(idx,N) = algtau;
            idx += N;
        }
    }
    
    // Done
    return res;
}

VectorXd LSCM::Xn::evall(LSCMint i) const {
    double h = (*grid)[i]-(*grid)[i-1];
    
    // Compute function values
    VectorXd algtau = vBasis->evalPa(1.0);
    VectorXd difftau = vBasis->evalPd(1.0);
    
    // Copy to final result
    VectorXd res(getdim());
    LSCMint idx = 0;
    for (LSCMint j = 0; j < m; ++j) {
        // Scaling from [0,1] to h appears here!
        if (D[j]) {
            res.segment(idx,N+1) = h*difftau;
            idx += N+1;
        }
        else {
            res.segment(idx,N) = algtau;
            idx += N;
        }
    }
    
    // Done
    return res;
}

VectorXd LSCM::Xn::deval(double t) const {
    // Compute function values
    LSCMint i = grid->interv(t)-1;
    double h = (*grid)[i+1]-(*grid)[i];
    double tau = (t-(*grid)[i])/h;
    
    return devalsub(tau,h);
}

VectorXd LSCM::Xn::devalsub(double tau, double h) const {
    
    VectorXd difftau = vBasis->evaldPd(tau);
    
    // Copy to final result
    VectorXd res = VectorXd::Zero(getdim());
    LSCMint idx = 0;
    for (LSCMint j = 0; j < m; j++) {
        if (D[j]) {
            res.segment(idx,N+1) = difftau;
            idx += N+1;
        }
        else
            idx += N;
    }
    
    // Done
    return res;
}
