/* File: WeightedSolver.hpp
 *
 * Realization of the weighted solver of linear DAEs
 * for a
 * given grid and approximation space.
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2022-06-05
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef WEIGHTEDSOLVER_HPP
#define WEIGHTEDSOLVER_HPP

#include "LSCMConfig.hpp"
#include "LinLSQSolver.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <Eigen/SPQRSupport>

namespace LSCM {

/**
 * Implementation of the standard version of the weighted solver
 * 
 * This class provides the weighted solver for the discrete system. The 
 * equations are ordered with the weighted equality constraints first.
 * 
 */
class WeightedSolver : public LinLSQSolver {
private:
    // weight for continuity conditions
    // There are different recommendations found in the literature:
    // Charles van Loan (1985): (macheps)^(-1/2)
    // Jesse L. Barlow (1992): (macheps)^(-1/3)
    double weight = std::pow(std::numeric_limits<double>::epsilon(),-1.0/3.0);
    
    // OBS: Using pointers to patch a bug in
    // ~Eigen::SPQR<Eigen::SparseMatrix<double>>()
    Eigen::SPQR<LSCMSparseMatrix>* QR = nullptr;
    LSCMint Arows, Crows, Acols;
    // Memory management for genmatT()
    LSCMindex *Xcolptr = nullptr;
    LSCMindex *Xrowidx = nullptr;
    double *Xvals = nullptr;
    
    void resetLocalVars() {
        delete QR;
        delete [] Xrowidx;
        delete [] Xcolptr;
        delete [] Xvals;
        QR = nullptr;
        Xrowidx = nullptr;
        Xcolptr = nullptr;
        Xvals = nullptr;
    }

    virtual void factorizeExec();
    
    virtual GridFkt solveExec(const Eigen::VectorXd& f);

public:
    /**
     * Default constructor
     */
    WeightedSolver() {}
    
    /**
     * Constructor for the weighted solver class
     * 
     * The weight is set to the value of \f$\epsilon_{\textrm{mach}}^{-1/3}\f$.
     * 
     * \param[in] genmat matrix generation object
     */
    WeightedSolver(std::shared_ptr<LinLSQMatrices> genmat) :
        LinLSQSolver(genmat) {}
    
    /**
     * Constructor for the weighted solver class
     * 
     * \param[in] genmat matrix generation object
     * \param[in] weight weight for the continuity conditions
     */
    WeightedSolver(std::shared_ptr<LinLSQMatrices> genmat, double wcont) :
    LinLSQSolver(genmat) {
        if (wcont <= 0.0) {
            std::cerr << "Error WeightedSolver: Positive weight required" << std::endl;
            exit(1);
        }
        else weight = wcont;
    }
    
    /**
     * Set the weight for the continuity conditions
     * 
     * @param[in] wpoLSCMint weight;
     */
    void setWeight(const double wcont) {
        if (wcont <= 0.0) {
            std::cerr << "Error setWeight: Positive weight required. Weight not changed" << std::endl;
        }
        else weight = wcont;
    }
    
    /**
     * Get the weight for the continuity conditions
     * 
     * @returns weight for continuity conditions
     */
    double getWeight() const { return weight; }

    ~WeightedSolver() {
        resetLocalVars();
    }
    
private:
    /**
     * Setup of system matrix
     */
    LSCMSparseMatrix genmatT();
};

}

#endif
