/* File: UserQuadrature.cpp
 *
 * User defined quadrature rule
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-14
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "UserQuadrature.hpp"
#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <cstdlib>
#include <limits>
#include <cmath>

using namespace LSCM;
using namespace Eigen;
using namespace std;

void UserQuadrature::generate() {
    // Check input data
    if (x(0) < -1.0 || x(n_-1) > 1.0) {
        cerr << "Error UserQuadrature: Nodes out of bounds." << endl;
        exit(1);
    }
    for (LSCMint i = 1; i < n_; ++i)
        if (x(i-1) >= x(i)) {
            cerr << "Error UserQuadrature: Nodes not monotone increasing." << endl;
            exit(1);
        }
    
    // Prepare using Legendre polynomials. Use recursion for stability
    MatrixXd V(n_,n_);
    VectorXd r = VectorXd::Zero(n_);
    
    V.row(0) = VectorXd::Constant(n_,1.0);
    r(0) = 2.0;
    if (n_ > 1) {
        V.row(1) = x.transpose();
        for (LSCMint i = 2; i < n_; i++) {
            auto tmp = x.transpose().cwiseProduct(V.row(i-1));
            V.row(i) = (((double) (i-1))/i)*(tmp-V.row(i-2))+tmp;
        }
    }

    // Solve
    w = V.colPivHouseholderQr().solve(r);
    // Check if the weights are reasonable
    for (LSCMint i = 0; i < n_; ++i)
        if (w(i) <= 0.0) {
            cerr << "Warning UserQuadrature: Negative weights. Method may be unusable." << endl;
        }

    // create barycentric weights
    // This procedure follows Berrut&Trefethen
    double C = 4.0/(x(n_-1)-x(0));
    c = VectorXd(n_);
    for (LSCMint i = 0; i < n_; ++i) {
        VectorXd cc = (VectorXd::Constant(n_,x(i))-x)*C;
        cc(i) = 1.0;
        c(i) = exp(-cc.array().abs().log().sum());
    }
    double cmax = c.maxCoeff();
    c *= (1.0/cmax);
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    baryScale = cmax/pow(C,n_-1);

    t = VectorXd::Zero(0);
    if (x(0) == -1.0) RuleProperties = QR_LEFT;
    if (x(n_-1) == 1.0) RuleProperties = RuleProperties | QR_RIGHT;
    if ((n_%2 == 1) && (x(n_/2) == 0.0)) RuleProperties = RuleProperties | QR_CENTERED;
    bool sym = true;
    for (LSCMint i = 0; i < n_/2; ++i)
        if (x(i) != -x(n_-i-1)) {
            sym = false;
            break;
        }
    if (sym) RuleProperties = RuleProperties | QR_SYMMETRIC;
}
