/* File: Radau.cpp
 *
 * Class implementing the Radau-Legendre integration
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Radau.hpp"
#include "LSCMConfig.hpp"
#include <iostream>
#include <cstdlib>

using namespace LSCM;
using namespace Eigen;
using namespace std;

// This implementation follows the ideas of chebfun
void Radau::generate() {
    VectorXd xt, wt, ct, tt;
    double dummy;
    Jacobi(n_-1,0.0,1.0,xt,wt,ct,tt,dummy);
    x = VectorXd(n_);
    x << -1.0, xt;

    ArrayXd tmp = ArrayXd::Constant(n_-1,1.0)/
                    (ArrayXd::Constant(n_-1,1.0)+xt.array());
    // Weights
    w = VectorXd(n_);
    w << 2.0/(n_*n_), wt.array()*tmp;

    // Barycentric weights (simplified)
    //c = VectorXd(n_);
    //c(0) = sqrt(2.0*w(0));
    //c.tail(n_-1) = sqrt((ArrayXd::Constant(n_-1,1.0)-xt.array())*wt.array());
    c = sqrt((ArrayXd::Constant(n_,1.0)-x.array())*w.array());
    double cmax = c.maxCoeff();
    c *= (1.0/cmax);
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    
    // Scale factor
    baryScale = bscaljac(n_-2,0.0,1.0)*cmax;
    
    // Orientation
    if (left_) RuleProperties = QR_LEFT;
    else {
        // Reverse
        VectorXd tmp(n_);
        tmp = x(seq(last,0,fix<-1>));
        x = -tmp;
        tmp = w(seq(last,0,fix<-1>));
        w = tmp;
        tmp = c(seq(last,0,fix<-1>));
        c = tmp;
        // Correct sign
        if (c(n_-1) < 0.0) c = -c;
        RuleProperties = QR_RIGHT;
    }
    
    t = VectorXd::Zero(0);
}
