/* File: RKCbasis.hpp
 *
 * This class implements the interface of class Basis using polynomials
 * in the Runga-Kutta representation. The basis functions are expressed
 * in terms of Legendre polynomials.
 * 
 * The interpolation points rho can either be given explicitely or by
 * providing their number. In the latter case, equidistantly distributed
 * grid points are used.
 * 
 * Note: Even if not strictly necessary, the interpolation points must be
 * strictly monotonic increasing.
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2020-04-06
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef RKCBASIS_HPP
#define RKCBASIS_HPP

#include "LSCMConfig.hpp"
#include "Basis.hpp"
#include <Eigen/Dense>
#include <vector>

namespace LSCM {

/**
 * Implementation of the Runge-Kutta basis for the DAE using a presentation
 * by Chebyshev polynomials
 * 
 * This class implements the interface of class Basis using polynomials
 * in the Runga-Kutta representation using Chebyshev polynomials.
 * 
 * The interpolation points rho can either be given explicitely or by
 * providing their number. In the latter case, Chebyshev nodes are used.
 */
class RKCbasis : public Basis {
private:
    // These matrices hold the polynomial coefficients for the functions as well
    // the derivatives. They can be used with Clenshaw scheme.
    Eigen::MatrixXd Va;  // Coefficents of the algebraic polynomial
    Eigen::MatrixXd Vd;  // Coefficients for the differential polynomial
    Eigen::MatrixXd dVd; // Coefficients for the derivative
    
    // Initialize Va, Vd, dVd
    void genbase(const std::vector<double>& rho);
    
    // Clenshaw scheme
    Eigen::VectorXd clenshaw(const Eigen::MatrixXd& V, double tau) const;
    
public:
    /**
     * Constructor for the Runge-Kutta basis
     * 
     * Here, the polynomial degree of the differential components N is
     * provided. The interpolation nodes are chosen to be the Chebyshev nodes
     * scaled to [0,1]
     * 
     * @param[in] N degree of the approximating differential polynomial
     */
    RKCbasis(LSCMint N);
    
    /**
     * Constructor for the Runge-Kutta basis
     * 
     * Here, the interpolation sequence rho is provided.
     * 
     * @param[in] rho interpolation nodes. They must belong to [0,1] and be
     * strictly monotonically increasing.
     */
    RKCbasis(const std::vector<double>& rho);
    
    Eigen::VectorXd evalPa(double tau) const override {
        return clenshaw(Va,tau); 
    }
    Eigen::VectorXd evalPd(double tau) const override {
        return clenshaw(Vd,tau);
    }
    Eigen::VectorXd evaldPd(double tau) const override {
        return clenshaw(dVd,tau);
    }
};

}

#endif
