/* File: RKLbasis.cpp
 *
 * This class implements the interface of class Basis using polynomials
 * in the Runga-Kutta representation using Legendre polynomials
 * 
 * The interpolation points rho can either be given explicitely or by
 * providing their number. In the latter case, equidistantly distributed
 * grid points are used.
 * 
 * Note: Even if not strictly necessary, the interpolation points must be
 * strictly monotonic increasing.
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2018-08-19
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#define _USE_MATH_DEFINES
#include "RKCbasis.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <cstdlib>
#include <cmath>
#include <vector>

using namespace std;
using namespace Eigen;

void LSCM::RKCbasis::genbase(const vector<double>& rho) {
    // Create the Lagrange base for the algebraic components
    // First: Vandermonde-like matrix
    MatrixXd V(dim,dim);
    for (LSCMint j = 0; j < dim; j++) V(0,j) = 1.0;
    if (dim > 1) {
        for (LSCMint j = 0; j < dim; ++j) {
            double cj = 2*rho[j]-1.0;
            V(1,j) = cj;
            for (LSCMint i = 2; i < dim; ++i) {
                V(i,j) = 2.0*cj*V(i-1,j)-V(i-2,j);
            }
        }
    }
    // Compute coefficients
    Va = V.inverse();
    
    // Create RK basis for the differential components including
    // its derivative
    Vd = MatrixXd(dim+1,dim+1);
    Vd(0,0) = 1.0;
    for (LSCMint j = 1; j <= dim; ++j) Vd(0,j) = 0.0;
    MatrixXd D = MatrixXd::Zero(dim,dim+1);
    D(0,0) = 0.5;
    D(0,1) = 0.5;
    if (dim > 1) {
        D(1,0) = -0.125;
        D(1,2) = 0.125;
        double fac = -0.5;
        for (LSCMint i = 2; i < dim; ++i) {
            D(i,i-1) = -0.25/(i-1);
            D(i,i+1) = 0.25/(i+1);
            D(i,0) = fac/(i*i-1);
            fac = -fac;
        }
    }
    Vd.bottomRows(dim) = Va*D;
    
    dVd = MatrixXd::Zero(dim+1,dim+1);
    dVd.bottomLeftCorner(dim,dim) = Va;
}

// Implements the Clenshaw scheme for a set of Legendre polynomials.
// The coefficients are the rows of V
// No check if tau is in [0,1]!
VectorXd LSCM::RKCbasis::clenshaw(const MatrixXd& V, double tau) const {
    LSCMint n = V.rows()-1;
    VectorXd z = V.col(n);
    VectorXd zold = VectorXd::Zero(n+1);
    double x = 2.0*tau-1.0;
    for (LSCMint k = n-1; k > 0; --k) {
        VectorXd ztmp = z;
        z = V.col(k)+(2.0*x)*z-zold;
        zold = ztmp;
    }
    if (n == 0) return z;
    return V.col(0)+x*z-zold;
}

LSCM::RKCbasis::RKCbasis(LSCMint N) : Basis(N) {
    // Create interpolation points: Chebyshev nodes
    double h = ((double) M_PI)/(2*dim);
    vector<double> rho(dim);
    for (LSCMint i = 0; i < dim; ++i) rho[i] = (1.0-cos((2*i+1)*h))*0.5;
    
    // Generate basis functions
    genbase(rho);
}

LSCM::RKCbasis::RKCbasis(const vector<double>& rho) : Basis(rho.size()) {
    // Check input
    for (LSCMint i = 1; i < dim; i++)
        if (rho[i-1] >= rho[i]) {
            cerr << "Error RKbasis: rho not monotone" << endl;
            exit(1);
        }
    if (rho[0] <= 0.0 || rho[dim-1] >= 1.0) {
        cerr << "Error RKbasis: rho out of bounds" << endl;
        exit(1);
    }
    
    // Generate basis functions
    genbase(rho);
}
