/* File: QuadratureRule.hpp
 *
 * Interface class for quadrature rules
 * 
 * C Michael Hanke 2023
 * Version: 2020-01-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef QUADRATURERULES_HPP
#define QUADRATURERULES_HPP

#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <vector>
#include <iostream>
#include <cstdlib>

namespace LSCM {

/**
 * Interface for quadrature rules
 * 
 * This class provides the interface for all quadrature rules. It shall be
 * used as a base class. Its inherited classes shall provide (at least)
 * integration nodes and weights as well as the barycentric weights.
 * If possible, more exact values for acos(x) can be provided.
 * 
 */
class QuadratureRule {
private:
    bool Dmatgenerated = false;
    
protected:
    LSCMint n_;  /**< Number of integration points. To be set in constructor */
    uint8_t RuleProperties = 0; /**< Properties of quadrature rules */
    Eigen::VectorXd x = Eigen::VectorXd::Zero(0);  /**< Integration nodes. Must be strictly increasing */
    Eigen::VectorXd w = Eigen::VectorXd::Zero(0);  /**< Integration weights */
    Eigen::VectorXd c = Eigen::VectorXd::Zero(0);  /**< Barycentric weights */
    Eigen::VectorXd t = Eigen::VectorXd::Zero(0);  /**< acos(x) */
    Eigen::MatrixXd Dmat; /**< Differentiation matrix */
    double baryScale = 1.0; /**< Scaling for the (simplified) barycentric weights */
    
public:
    /**
     * Structure holding the nodes and weights for integration formulae
     */
    typedef struct { double zero, weight; } IntegrationZW;
    
    /**
     * Vector of (nodes,weight)-pairs for numerical intergartion rules
     * 
     * For a given function \f$f\f$, the integral \f$I = \int_0^1 f(\tau)d\tau\f$
     * can be approximated by
     * \f[
     *    I\approx \sum_{i=0}^M \textsf{IntMethod}[i].\textsf{weight}*
     *    f(\textsf{IntMethod}[i].\textsf{zero})
     * \f]
     * where M = IntMethod.size().
     */
    typedef std::vector<IntegrationZW> IntMethod;

    /**
     * Constants for properties of the integration rule
     */
    static const uint8_t QR_CENTERED = 1;  /**<  0 is a node */
    static const uint8_t QR_LEFT = 2;      /**< -1 is a node */
    static const uint8_t QR_RIGHT = 4;     /**< +1 is a node */
    static const uint8_t QR_SYMMETRIC = 8; /**<  symmetric formula */
    
private:
    void genDmat();
    
protected:
    /**
     * Service routine. It provides the nodes and weights for the
     * Gauss-Jacobi integration
     * 
     * \param[in]  n     number of integration nodes
     * \param[in]  alpha exponent of weight function (1-x)
     * \param[in]  beta  exponent of weight function (1+x)
     * \param[out] x     integration nodes
     * \param[out] w     integration weights
     * \param[out] c     barycentric weights
     * \param[out] scale scale of the barycentric weights
     */
    void Jacobi(const LSCMint n, const double alpha, const double beta,
                Eigen::VectorXd& x,
                Eigen::VectorXd& w, Eigen::VectorXd& c, Eigen::VectorXd& t,
                double& scale);
    
    /**
     * Service routine. Computation of the scaling factor for simplified barycentric
     * weights for Jacobi polynomials. Adapted from Wang, Huybrechts, Vandewalle
     * 
     * \param[in] n
     * \param[in] alpha
     * \param[in] beta
     * \returns scaling factor
     */
    double bscaljac(const LSCMint n, const double alpha, const double beta);
    
    /**
     * Generate the required data (Dmat not required). jacpts can be used if
     * appropriate
     */
    virtual void generate() = 0;
    
    /**
     * Constructor for special cases: User quadrature
     * 
     * \param[in] tau vector of integration nodes. It must hold
     *                -1 <= tau(0) < ... < tau(end) <= 1.
     */
    QuadratureRule(Eigen::VectorXd tau) : n_(tau.size()), x(tau) {}
    
public:
    /**
     * Constructor 
     */
    QuadratureRule() = delete;  // no default constructor
    
    /** Constructor
     * 
     * \param[in] n number of integration nodes
     */
    QuadratureRule(LSCMint n) : n_(n) {
        if (n <= 0) {
            std::cerr << "Error QuadratureRule: n must be positive!" << std::endl;
            std::exit(1);
        }
        if (n > 100)
            std::cerr << "Warning QuadratureRule: n > 100. Results may be unreliable." << std::endl;
    }
    
    /**
     * Provides the integration nodes
     * 
     * \returns integration nodes
     */
    const Eigen::VectorXd& getnodes() {
        if (w.size() == 0) generate();
        return x;
    }
    
    /**
     * Provides the integration weights
     * 
     * \returns integration weights
     */
    const Eigen::VectorXd& getweights() {
        if (w.size() == 0) generate();
        return w;
    }
    
    /**
     * Provides the scaled barycentric weights
     * 
     * \returns berycentric weights
     */
    const Eigen::VectorXd& getbarywts() {
        if (w.size() == 0) generate();
        return c;
    }
    
    /**
     * Provides the inverse cosine of the integration nodes
     * 
     * \returns acos(integration nodes)
     */
    const Eigen::VectorXd& getcosargs() {
        if (w.size() == 0) generate();
        return t;
    }
    
    /**
     * Provides the differentiation matrix
     * 
     * \returns differentiation matrix
     */
    const Eigen::MatrixXd& getDifferentiationMatrix() {
        if (w.size() == 0) generate();
        if (!Dmatgenerated) genDmat();
        // Apply null sum trick
        for (LSCMint i = 0; i < Dmat.rows(); ++i) {
            double sum = 0.0;
            for (LSCMint j = 0; j < Dmat.cols(); ++j)
                if (i != j) sum += Dmat(i,j);
            Dmat(i,i) = -sum;
        }
        return Dmat;
    }
    
    /**
     * Provides the quadrature rule's properties
     * 
     * \returns rule properties
     */
    uint8_t getProperties() {
        if (w.size() == 0) generate();
        return RuleProperties;
    }
    
    /**
     * Returns the scaling facor for the barycentric weights
     * 
     * The barycentric weights are usually computed in a scaled fashion in order
     * to avoid over- and underflow. The scaling factor can be used to obtain the
     * exact barycentric weights on [-1,1].
     * 
     * \returns scaling factor
     */
    double getbaryscale() {
        if (w.size() == 0) generate();
        return baryScale;
    }
    
    /**
     * Compatibility function for LSCM
     * 
     * \returns integration nodes and weights scaled to [0,1]
     */
    IntMethod getMethod() {
        if (w.size() == 0) generate();
        IntMethod meth(n_);
        for (LSCMint i = 0; i < n_; ++i) {
            meth[i].zero = 0.5*(x(i)+1.0);
            meth[i].weight = 0.5*w(i);
        }
        return meth;
    }
    
    /**
     * Compatibility function for LSCM
     * 
     * "Square root" of mass matrix for L^2-norm approximation scales to (0,1)
     * 
     * For a given function \f$f\f$, let \f$Rf\f$ denote the polynomial interpolation using
     * the nodes tau. Then, the integral \f$I=\int_0^1 (Rf)^2(\tau)d\tau\f$
     * can be approximated by
     * \f[
     *    I\approx f_d^T A^T A f_d
     * \f]
     * where \f$f_d\f$ is the vector of node values of \f$f\f$.
     * 
     *  @returns   matrix A
     */
    Eigen::MatrixXd gen2mass();
    
    virtual ~QuadratureRule() = default;
};

}

#endif
