/* File: NullspaceSolver.hpp
 *
 * Realization of the direct solver of linear DAEs
 * for a
 * given grid and approximation space. Implementation follows Björck
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2022-06-06
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef NULLSPACESOLVER_HPP
#define NULLSPACESOLVER_HPP

#include "LSCMConfig.hpp"
#include "LinLSQSolver.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <Eigen/SPQRSupport>
#include <Eigen/OrderingMethods>

namespace LSCM {

// NOTE: SparseQR must be used since Q not available in SPQR!

/**
 * Implementation of the Björck version of the nullspace solver
 * 
 * This class provides another direct solver for the discrete system.
 * NOTE: This method requires a lot of memory. Use it only if no other method provides
 * is applicable.
 * 
 */
class NullspaceSolver : public LinLSQSolver {
private:
    Eigen::SPQR<LSCMSparseMatrix>* qrAQ2 = nullptr;
    Eigen::PermutationMatrix<Eigen::Dynamic,Eigen::Dynamic,LSCMindex> P;
    LSCMSparseMatrix Q1,Q2;
    LSCMSparseMatrix RBT;
    LSCMSparseMatrix A;
    LSCMint Arow;
    LSCMint Crow;
    
    void resetLocalVars() {
        delete qrAQ2;
        qrAQ2 = nullptr;
    }

    virtual void factorizeExec();
    
    virtual GridFkt solveExec(const Eigen::VectorXd& f);

public:
    /**
     * Default constructor
     */
    NullspaceSolver() {}
    
    /**
     * Constructor for the direct solver class
     * 
     * \param[in] genmat matrix generation object
     */
    NullspaceSolver(std::shared_ptr<LinLSQMatrices> genmat) :
        LinLSQSolver(genmat) {}
    
    ~NullspaceSolver() {
        resetLocalVars();
    }
};

}

#endif
