/* File: NullspaceSolver.cpp
 *
 * Realization of the direct solver of linear DAEs
 * for a
 * given grid and approximation space.  Implementation follows Björck
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2022-06-06
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "LSCMConfig.hpp"
#include "NullspaceSolver.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <Eigen/SPQRSupport>
#include <ctime>

using namespace LSCM;
using namespace Eigen;
using namespace std;

// This function is a workaround for a deficiency LSCMint Eigen's SPQR interface.
namespace {
LSCMSparseMatrix
SPQRtoQ(const SPQR<LSCMSparseMatrix>& qrC, LSCMint istart, LSCMint col, LSCMint dim)
{
    vector<Triplet<double>> Qlist;
    Qlist.reserve(dim*col/10); // Factor from experiments
    VectorXd sj = VectorXd::Zero(dim);
    for (LSCMint j = 0; j < col; ++j) {
        sj(istart+j) = 1.0;
        VectorXd tqj = qrC.matrixQ()*sj;
        sj(istart+j) = 0.0;
        for (LSCMint i = 0; i < dim; ++i)
            if (tqj(i) != 0.0)
                Qlist.push_back(Triplet<double>(i,j,tqj(i)));
    }
    LSCMSparseMatrix Q(dim,col);
    Q.setFromTriplets(Qlist.begin(),Qlist.end());

    return Q;
}
}

void LSCM::NullspaceSolver::factorizeExec()
{
    clock_t tass = clock();
    
    A = mat->genA();
    LSCMSparseMatrix C = (mat->genC()).transpose(); // Not really efficient!

    timings.tass = ((double)(clock()-tass))/CLOCKS_PER_SEC;
    
    Crow = C.cols();
    LSCMint nun = C.rows();
    Arow = A.rows();
    
    mem.nnzA = A.nonZeros();
    mem.dimA = Arow;
    mem.nnzC = C.nonZeros();
    mem.dimC = Crow;
    mem.ndim = nun;
    
    clock_t tfact = clock();
   
    // The algorithm requires too much memory in this implementation!
    SPQR<LSCMSparseMatrix> qrC;
    qrC.compute(C);
    if (qrC.info() != Success) {
        // cerr << QR.lastErrorMessage() << endl;
        cerr << "NullspaceSolver: QR decomposition failed" << endl;
        exit(1);
    }
    RBT = qrC.matrixR().transpose().leftCols(Crow);
    // Check fullrank: Always true!
    for (LSCMint i = 0; i < Crow; ++i)
        if (RBT.coeff(i,i) == 0.0) {
            cerr << "DirectSolver: Continuity conditions dependent!" << endl;
            exit(1);
        }
    
    P = qrC.colsPermutation();
    // Since C is very sparse and regular, we expect Q to be very sparse, too. Our
    // experiments confirm this. Therefore, we compute Q1 and Q2 explicitely.
    //
    // The following procedure is a workaround for a deficiency in Eigen's SPQR
    // interface: SuiteSparseQR_qmult() for sparse matrix right-hand sides is not
    // accessible in Eigen.
    // The following procedure requires too much memory!
    // NOTE: Must use a full S2!!!
    //MatrixXd S2 = MatrixXd::Zero(nun,nun-Crow);
    //S2.bottomRows(nun-Crow).setIdentity(nun-Crow,nun-Crow);
    //Q2 = (qrC.matrixQ()*S2).eval().sparseView();
    // Therefore: Do it by hand instead :-(
    Q2 = SPQRtoQ(qrC,Crow,nun-Crow,nun);
    // Now the same strange thing for Q1 if necessary
    if (!homConstraints) Q1 = SPQRtoQ(qrC,0,Crow,nun);
    
    qrAQ2 = new SPQR<LSCMSparseMatrix>;
    qrAQ2->compute(A*Q2);
    if (qrAQ2->info() != Success) {
        // cerr << QR.lastErrorMessage() << endl;
        cerr << "NullspaceSolver: QR decomposition failed" << endl;
        exit(1);
    }
    
    timings.tfact = ((double)(clock()-tfact))/CLOCKS_PER_SEC;
    
    // Check statistics
    cholmod_common *cc = qrC.cholmodCommon();
    cholmod_common *cx = qrAQ2->cholmodCommon();
    mem.nWork =
        cc->SPQR_istat[0]+   // cc->SPQR_istat[1] not needed (but stored in this implementation
        cx->SPQR_istat[0]+
        cx->SPQR_istat[1]+
        Q2.nonZeros();   // Obs: nnz(A*Q2) missing!
        // cerr << "+++ H nnz: " << cx->SPQR_istat[1] << endl;

    DecompositionAvail = true;
}

GridFkt LSCM::NullspaceSolver::solveExec(const VectorXd& f)
{
    clock_t tslv = clock();
    
    VectorXd tmpCoeffs;
    VectorXd SolCoeffs;
    if (homConstraints) {
        tmpCoeffs = qrAQ2->solve(f);
        if (qrAQ2->info() != Success) {
            // cerr << QR.lastErrorMessage() << endl;
            cerr << "NullspaceSolver: solve failed" << endl;
            exit(1);
        }
        SolCoeffs = Q2*tmpCoeffs;
    }
    else {
        VectorXd b = VectorXd::Zero(Crow);
        b.head(numBC) = f.tail(numBC);
        VectorXd x1 = Q1*RBT.triangularView<Lower>().solve(P.transpose()*b);
        tmpCoeffs = qrAQ2->solve((f.head(Arow)-A*x1).eval());
        if (qrAQ2->info() != Success) {
            // cerr << QR.lastErrorMessage() << endl;
            cerr << "NullspaceSolver: solve failed" << endl;
            exit(1);
        }
        SolCoeffs = Q2*tmpCoeffs+x1;
    }
    
   timings.tslv = ((double)(clock()-tslv))/CLOCKS_PER_SEC;
    
    return GridFkt(mat->getspace(),SolCoeffs);
}

