/* File: NlDAE.hpp
 *
 * This class collects the functions describing a nonlinear DAE.
 *
 * The definition of the grid and the approximating space depends already
 * on the DAE. 
 *
 * Copyright (C) Michael Hanke 2019
 * Version: 2018-12-17
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef NLDAE_HPP
#define NLDAE_HPP

#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <functional>
#include <vector>

namespace LSCM {

/**
 * This class describes the interface to a nonlinear DAE.
 * 
 * We use the model
 * \f[
 *   f((Dx)'(t),x(t),t)=0,\quad r(x(a),x(b))=0
 * \f]
 * for \f$t\in(a,b)\f$.
 *
 * Note that The definition of the grid and the approximating space depends already
 * on the DAE!
 */
class NlDAE {
protected:
    LSCMint l = 0;
    LSCMint index = -1;
public:
    /**
     * Type describing the interface to DAE functions
     */
    typedef std::function<Eigen::VectorXd(const Eigen::VectorXd&,
                               const Eigen::VectorXd&,
                               const double)> DAEfunc;
    
    /**
     * Type describing the interface to the derivatives
     * of the DAE functions
     */
    typedef std::function<Eigen::MatrixXd(const Eigen::VectorXd&,
                               const Eigen::VectorXd&,
                               const double)> DAEdfunc;
    
    /**
     * Type describing the interface to the boundary
     * functions
     */
    typedef std::function<Eigen::VectorXd(const Eigen::VectorXd&,
                                const Eigen::VectorXd&)> RBfunc;
    
    /**
     * Type describing the interface to the derivatives
     * of the boundary functions
     */
    typedef std::function<Eigen::MatrixXd(const Eigen::VectorXd&,
                                const Eigen::VectorXd&)> RBdfunc;
    
    /**
     * The function f
     * 
     * \param[in] y argument (Dx)'(t)
     * \param[in] x argument x(t)
     * \param[in] t poLSCMint in (a,b)
     * \returns f(y,x,t)
     */
    virtual Eigen::VectorXd f(const Eigen::VectorXd& y,
                              const Eigen::VectorXd& x,
                              const double t) = 0;
    
    /**
     * The partial derivative of f with respect to y
     * 
     * \param[in] y argument (Dx)'(t)
     * \param[in] x argument x(t)
     * \param[in] t poLSCMint in (a,b)
     * \returns \f$f_y(y,x,t)\f$
     */
    virtual Eigen::MatrixXd dfy(const Eigen::VectorXd& y,
                                const Eigen::VectorXd& x,
                                const double t) = 0;
    
    /**
     * The partial derivative of f with respect to x
     * 
     * \param[in] y argument (Dx)'(t)
     * \param[in] x argument x(t)
     * \param[in] t poLSCMint in (a,b)
     * \returns \f$f_x(y,x,t)\f$
     */
    virtual Eigen::MatrixXd dfx(const Eigen::VectorXd& y,
                                const Eigen::VectorXd& x,
                                const double t) = 0;

    /**
     * The function r
     * 
     * \param[in] xa argument x(a)
     * \param[in] xb argument x(b)
     * \returns r(xa,xb)
     */
    virtual Eigen::VectorXd r(const Eigen::VectorXd& xa,
                              const Eigen::VectorXd& xb) = 0;
    
    /**
     * The partial derivative of r with respect to its first argument
     * 
     * \param[in] xa argument x(a)
     * \param[in] xb argument x(b)
     * \returns \f$r_{x(a)}(xa,xb)\f$
     */
    virtual Eigen::MatrixXd dra(const Eigen::VectorXd& xa,
                                const Eigen::VectorXd& xb) = 0;
    
    /**
     * The partial derivative of r with respect to its second argument
     * 
     * \param[in] xa argument x(a)
     * \param[in] xb argument x(b)
     * \returns \f$r_{x(b)}(xa,xb)\f$
     */
    virtual Eigen::MatrixXd drb(const Eigen::VectorXd& xa,
                                const Eigen::VectorXd& xb) = 0;
    
    /**
     * The D coefficient
     * 
     * \returns diag(D)
     */
    virtual std::vector<bool> getD() = 0;
    
    /**
     * Returns number of boundary conditios
     * 
     * @returns l
     */
    virtual LSCMint getl() { return l; } 
    
    /*
     * returns the index of the problem (if known)
     * 
     * @returns index
     */
    LSCMint getindex() { return index; }
    
    virtual ~NlDAE() = default;
   
};

}

#endif
