/* File: Lobattos.cpp
 *
 * Class implementing the Lobatto-Legendre integration
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Lobatto.hpp"
#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <cstdlib>
#include <limits>

using namespace LSCM;
using namespace Eigen;
using namespace std;

namespace {
    const double besselzeros[] = {
     3.831705970207512e+00,
     7.015586669815619e+00,
     1.017346813506272e+01,
     1.332369193631422e+01,
     1.647063005087763e+01,
     1.961585851046824e+01,
     2.276008438059277e+01,
     2.590367208761838e+01,
     2.904682853491686e+01,
     3.218967991097441e+01,
     3.533230755008387e+01,
     3.847476623477161e+01,
     4.161709421281445e+01,
     4.475931899765282e+01,
     4.790146088718545e+01,
     5.104353518357151e+01,
     5.418555364106132e+01,
     5.732752543790101e+01,
     6.046945784534749e+01,
     6.361135669848123e+01,
     6.675322673409849e+01,
     6.989507183749578e+01,
     7.303689522557383e+01,
     7.617869958464146e+01,
     7.932048717547630e+01,
     8.246225991437356e+01,
     8.560401943635023e+01,
     8.874576714492630e+01,
     9.188750425169499e+01,
     9.502923180804470e+01,
     9.817095073079078e+01,
     1.013126618230387e+02,
     1.044543657912828e+02,
     1.075960632595092e+02,
     1.107377547808992e+02,
     1.138794408475950e+02,
     1.170211218988924e+02,
     1.201627983281490e+02,
     1.233044704886357e+02,
     1.264461386985166e+02,
     1.295878032451040e+02,
     1.327294643885096e+02,
     1.358711223647890e+02,
     1.390127773886597e+02,
     1.421544296558590e+02,
     1.452960793451959e+02,
     1.484377266203422e+02,
     1.515793716314014e+02,
     1.547210145162859e+02,
     1.578626554019303e+02,
     1.610042944053620e+02,
     1.641459316346496e+02,
     1.672875671897441e+02,
     1.704292011632266e+02,
     1.735708336409759e+02,
     1.767124647027638e+02,
     1.798540944227884e+02,
     1.829957228701530e+02,
     1.861373501092955e+02,
     1.892789762003760e+02,
     1.924206011996257e+02,
     1.955622251596626e+02,
     1.987038481297770e+02,
     2.018454701561909e+02,
     2.049870912822923e+02,
     2.081287115488500e+02,
     2.112703309942078e+02,
     2.144119496544620e+02,
     2.175535675636242e+02,
     2.206951847537694e+02,
     2.238368012551717e+02,
     2.269784170964295e+02,
     2.301200323045791e+02,
     2.332616469052006e+02,
     2.364032609225143e+02,
     2.395448743794699e+02,
     2.426864872978287e+02,
     2.458280996982398e+02,
     2.489697116003099e+02,
     2.521113230226686e+02,
     2.552529339830281e+02,
     2.583945444982395e+02,
     2.615361545843440e+02,
     2.646777642566215e+02,
     2.678193735296346e+02,
     2.709609824172707e+02,
     2.741025909327807e+02,
     2.772441990888146e+02,
     2.803858068974556e+02,
     2.835274143702514e+02,
     2.866690215182434e+02,
     2.898106283519944e+02,
     2.929522348816139e+02,
     2.960938411167825e+02,
     2.992354470667742e+02,
     3.023770527404775e+02,
     3.055186581464156e+02,
     3.086602632927644e+02,
     3.118018681873705e+02,
     3.149434728377672e+02,
     3.180850772511903e+02,
     3.212266814345928e+02,
     3.243682853946578e+02,
     3.275098891378125e+02,
     3.306514926702394e+02,
     3.337930959978885e+02,
     3.369346991264880e+02,
     3.400763020615541e+02,
     3.432179048084012e+02,
     3.463595073721510e+02,
     3.495011097577409e+02,
     3.526427119699324e+02,
     3.557843140133188e+02,
     3.589259158923327e+02,
     3.620675176112526e+02,
     3.652091191742101e+02,
     3.683507205851957e+02,
     3.714923218480648e+02,
     3.746339229665437e+02,
     3.777755239442346e+02,
     3.809171247846209e+02,
     3.840587254910721e+02,
     3.872003260668482e+02,
     3.903419265151044e+02,
     3.934835268388949e+02,
     3.966251270411776e+02,
     3.997667271248168e+02,
     4.029083270925880e+02
    };
    
    const LSCM::LSCMint besselsize = sizeof(besselzeros)/sizeof(besselzeros[0]);
}

void Lobatto::genlobatto() {
    VectorXd xt, wt, ct, tt;
    double dummy;
    Jacobi(n_-2,1.0,1.0,xt,wt,ct,tt,dummy);

    x = VectorXd(n_);
    x << -1.0, xt, 1.0;
        
    w = VectorXd(n_);
    w << 2.0/(n_*(n_-1)), wt.array()/
            (ArrayXd::Constant(n_-2,1.0)-xt.cwiseProduct(xt).array()),
                2.0/(n_*(n_-1));
        
    c = w.array().sqrt();
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    baryScale = bscaljac(n_-3,1.0,1.0);
    t = VectorXd::Zero(0);
}

void Lobatto::generate() {
    // Technical constants
    const double pi2 = M_PI*M_PI;
    const double tol = 1e1*numeric_limits<double>::epsilon();
    const LSCMint itmax = 100;
    const LSCMint Mmax = besselsize*2+3;

    // trival cases
    if (n_ > Mmax) {
        // Fallback. 
        // use jacpts
        genlobatto();
        return;
    }
    if (n_ == 2) {
        x = VectorXd::Zero(2);
        x(0) = -1.0;
        x(1) = 1.0;
        w = VectorXd::Constant(2,1.0);
        c = VectorXd::Zero(2);
        c(0) = -0.5;
        c(1) = 0.5;
        t = VectorXd::Zero(2);
        t(0) = M_PI;
        baryScale = 1.0;
    
        // Set properties
        RuleProperties = QR_LEFT | QR_RIGHT | QR_SYMMETRIC;
        return;
    }
    if (n_ == 3) {
        x = VectorXd::Zero(3);
        x(0) = -1.0;
        x(2) = 1.0;
        w = VectorXd::Constant(3,1/3);
        w(1) += 1.0;
        c = VectorXd::Zero(3);
        c(0) = 0.5;
        c(1) = -1.0;
        c(2) = 0.5;
        t = VectorXd::Zero(3);
        t(0) = M_PI;
        t(1) = 0.5*M_PI;
        baryScale = 1.0;
    
        // Set properties
        RuleProperties = QR_LEFT | QR_RIGHT | QR_SYMMETRIC | QR_CENTERED;
        return;
    }
        
    // This an implementation of Michels' algorithm
    LSCMint m = n_/2-1;  // integer division!
    VectorXd xt(m);
    for (LSCMint i = 0; i < m; ++i)
        xt(i) = cos(besselzeros[i]/
                sqrt((n_-0.5)*(n_-0.5)+0.25*(pi2-4.0)/pi2));
    LSCMint it = 0;
    VectorXd z = VectorXd::Constant(m,1.0);
    while ((z.lpNorm<Infinity>() >= tol) && (it <= itmax)) {
        // Compute P_{M-1}' and P_{M-1}''
        // Note: M > 3 here!
        VectorXd pm1m2 = VectorXd::Constant(m,1.0);
        VectorXd pm1m1 = 3.0*xt;
        VectorXd pm2m2 = VectorXd::Zero(m);
        VectorXd pm2m1 = VectorXd::Constant(m,3.0);
        for (LSCMint j = 3; j < n_; ++j) {
            VectorXd tmp1 = xt.cwiseProduct(pm1m1);
            VectorXd pm1 = (((double) j)/(j-1))*(tmp1-pm1m2)+tmp1;
            pm1m2 = pm1m1;
            pm1m1 = pm1;
            VectorXd tmp2 = xt.cwiseProduct(pm2m1);
            VectorXd pm2 = (((double) (j+1))/(j-2))*(tmp2-pm2m2)+tmp2;
            pm2m2 = pm2m1;
            pm2m1 = pm2;
        }
        z = pm1m1.array()/pm2m1.array();
        xt -= z;
        ++it;
    }
    if (it > itmax) {
        cerr << "Error Lobatto: Node computation failed" << endl;
        exit(1);
    }
    
    // The nodes so far appear in reverse order!
    x = VectorXd(n_);
    for (LSCMint i = 1; i <= m; ++i) {
        x(i) = -xt(i-1);
        x(n_-i-1) = xt(i-1);
    }
    x(0) = -1.0;
    x(n_-1) = 1.0;
    if ( n_%2 == 1) x(m+1) = 0.0;

    // Security checks
    // Monotonicy
    for (LSCMint i = 1; i < n_; ++i) {
        if (x(i) <= x(i-1)) {
            cerr << "Error Lobatto: Something went wrong" << endl;
            exit(1);
        }
    }
    // really different zeros
    for (LSCMint i = 1; i < n_; ++i) {
        if (abs(x(i)-x(i-1)) < 1e1*tol) {
            cerr << "Error Lobatto: Something went wrong" << endl;
            exit(1);
        }
    }
        
    // final step: Optimized using the weights' symmetry
    LSCMint m1 = (n_+1)/2;
    VectorXd pm1m2 = VectorXd::Constant(m1,1.0);
    VectorXd pm1m1 = x.tail(m1);
    for (LSCMint j = 2; j < n_; ++j) {
        VectorXd tmp = x.tail(m1).cwiseProduct(pm1m1);
        VectorXd pm1 = (((double) j-1)/j)*(tmp-pm1m2)+tmp;
        pm1m2 = pm1m1;
        pm1m1 = pm1;
    }
    c = VectorXd(n_);
    double fac = sqrt(2.0/(n_*(n_-1)));
    c.tail(m1) = (ArrayXd::Constant(m1,fac)/pm1m1.array()).abs();
    for (LSCMint i = 0; i <= m; ++i) c(i) = c(n_-i-1);
    w = c.cwiseProduct(c);
    double cmax = c.maxCoeff();
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    c *= (1.0/cmax);

    t = VectorXd::Zero(0);
    baryScale = bscaljac(n_-3,1.0,1.0)*cmax;
    
    // Set properties
    RuleProperties = QR_LEFT | QR_RIGHT | QR_SYMMETRIC;
    if (n_%2 == 1) RuleProperties = RuleProperties | QR_CENTERED;
}
