/* File: LinearizedDAE.hpp
 *
 * This class defines the linearization of a nonlinear dae
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2019-12-17
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LINEARIZEDDAE_HPP
#define LINEARIZEDDAE_HPP

#include "LSCMConfig.hpp"
#include "DAE.hpp"
#include "NlDAE.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <memory>

namespace LSCM {

/**
 * This class provides the linearization of a nonlinear DAE for using in
 * an iterative solver
 */
class LinearizedDAE : public DAE {
private:
    NlDAE* daeobj;
    const Eigen::VectorXd xa;
    const Eigen::VectorXd xb;
    const Eigen::VectorXd r;
    const Eigen::MatrixXd Ga;
    const Eigen::MatrixXd Gb;
    const GridFkt* xp;
public:
    /**
     * Construction of the linear function given a nonlinear DAE and the
     * linarization point
     * 
     * @param[in] dae desription of a nonlinear dae
     * @param[in] x linarization point
     */
    LinearizedDAE(std::shared_ptr<NlDAE> dae, const GridFkt& x) : daeobj(dae.get()), xp(&x),
        xa(x.eval(x.geta())),
        xb(x.eval(x.getb())),
        r(dae->r(xa,xb)),
        Ga(dae->dra(xa,xb)),
        Gb(dae->drb(xa,xb)) {
            l = daeobj->getl(); 
            index = daeobj->getindex();
    }
    
    /**
     * Coefficient in front of the derivative
     * 
     * @param[in] t evaluation point
     * @returns the coefficient matrix evaluated at t
     */
    Eigen::MatrixXd A(double t) {
        Eigen::VectorXd yt = xp->deval(t);
        Eigen::VectorXd xt = xp->eval(t);
        return daeobj->dfy(yt,xt,t);
    }
    
    /**
     * Coefficient in front of the function
     * 
     * @param[in] t evaluation point
     * @returns the coefficient matrix evaluated at t
     */
    Eigen::MatrixXd B(double t) {
        Eigen::VectorXd yt = xp->deval(t);
        Eigen::VectorXd xt = xp->eval(t);
        return daeobj->dfx(yt,xt,t);
    }
    
    /**
     * Right-hand side
     * 
     * @param[in] t evaluation point
     * @returns the right-hand side evaluated at t
     */
    Eigen::VectorXd q(double t) {
        Eigen::VectorXd yt = xp->deval(t);
        Eigen::VectorXd xt = xp->eval(t);
        return daeobj->f(yt,xt,t);
    }
    
    /**
     * @returns the decription of D
     */
    std::vector<bool> getD() { return daeobj->getD(); }
    
    /**
     * @returns boundary matrix at a
     */
    Eigen::MatrixXd getGa() { return Ga; }
    
    /**
     * @returns boundary matrix at b
     */
    Eigen::MatrixXd getGb() { return Gb; }
    
    /**
     * @returns right-hand side of boundary condition
     */
    Eigen::VectorXd getr() { return r; }
};

}

#endif

