/* File: LinLSQStepper.hpp
 *
 * Implements a time stepper for linear DAEs
 * 
 * C Michael Hanke 2023
 * Version: 2023-05-15
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LINLSQSTEPPER_HPP
#define LINLSQSTEPPER_HPP

#include "LSCMConfig.hpp"
#include "DAE.hpp"
#include "GridFkt.hpp"
#include "Grid.hpp"
#include "Basis.hpp"
#include "LinLSQSolver.hpp"
#include "Yn.hpp"
#include "Eigen/Dense"
#include <memory>
#include <deque>
#include <cstdlib>

namespace LSCM {

/**
 * Implementation of a time stepper for linear DAEs
 * 
 * Time stepper for linear DAEs. In this implementation,
 * the grid is given a-priori and not adapted until a sound error
 * estimator is available.
 */
class LinLSQStepper {
private:
    // Discretization determining variables
    const std::shared_ptr<DAE> dae_;
    const std::shared_ptr<Basis> basis_;
    const std::shared_ptr<Yn> colloc_;
    std::shared_ptr<LinLSQSolver> slv_;
    
public:
    /**
     * This type contains the timings for the linear time stepper.
     * Unit is seconds
     */
    typedef struct {
        double tsub = 0.0;  /**> time for the LSQ solves */
        double tiv = 0.0;   /**> time for the initialization */
    } StepperTimings;
    
private:
    StepperTimings timings;
    
public:
    /**
     * Constructor
     * 
     * The initial value problem is described by an intance of the class
     * DAE. The latter is intended for boundary value problems with boundary
     * conditions of the form
     * \f[ G_a x(a)+G_b x(b) = r \f]
     * with \f$G_a\f$ and \f$G_b\f$ having identical dimansions. Therefore,
     * the dae object is required to provide \f$G_b=0\f$!
     * The variable BCversion selects the handling of initial conditions:
     * If being true, the least-squares approach is used while for false,
     * the initial conditions are handled as equality constraints.
     * 
     * @param[in] dae   dae to be solved. Must be an initial value problem!
     * @param[in] basis approximation space
     * @param[in] colloc collocation criterion
     * @param[in] slv   Linear solver
     */
    LinLSQStepper(const std::shared_ptr<DAE> dae, 
              const std::shared_ptr<Basis> basis,
              const std::shared_ptr<Yn> colloc,
              std::shared_ptr<LinLSQSolver> slv) : 
              dae_(dae), basis_(basis), colloc_(colloc), slv_(slv) {}
    
    /**
     * Time stepper for linear DAEs. In this implementation,
     * the grid is given a-priori and not adapted.
     * 
     * @param[in] grid  basic grid
     * @param[in] micro number of (uniform) micro steps in each subinterval
     *                  of the grid
     * \param[in] checkIC check if the provided initial condition is
     *                    accurate (true). Default is false
     * @param[in] BCversion How to handle initial conditions: as part of the
     *            functional (true) or as equality constraint (false)
     * @returns   Pointer to vector of local solutions. The i-th element
     *            contains the approximation on the i-th subinterval of the
     *            chosen grid
     */
    std::shared_ptr<std::deque<GridFkt>> solve(std::shared_ptr<Grid> grid,
                                               LSCMint micro,
                                               bool checkIC = false,
                                               bool BCversion = true);
    
    /**
     * Returns timings for the last solution process
     */
    StepperTimings gettimings() const { return timings; }

};

}

#endif
