/* File: LinLSQSolver.cpp
 *
 * This class is intended to approximate the solution to linear DAEs for a
 * given grid and approximation space. Some auxiliary routines are made
 * public for testing purposes. However, the main function is solve.
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2022-06-09
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "LinLSQSolver.hpp"
#include "GridFkt.hpp"
#include "Xn.hpp"
#include "Eigen/Dense"

using namespace LSCM;
using namespace Eigen;
using namespace std;

void LSCM::LinLSQSolver::factorize1i()
{
    MatrixXd X = mat->genA1i();
    QR.compute(X);
    DecompositionAvail = true;
}

GridFkt LSCM::LinLSQSolver::solve1i(const VectorXd& f)
{
    if (!DecompositionAvail) factorize1i();
    
    VectorXd SolCoeffs = QR.solve(f);
    return GridFkt(mat->getspace(),SolCoeffs);
}

void LSCM::LinLSQSolver::factorize()
{
    if (mat == nullptr) {
        cerr << "LinLSQSolver: No DAE set!" << endl;
        exit(1);
    }

    if (DecompositionAvail) return;
    
    oneD = (mat->getspace()->getgrid()->getn() == 1) &&
    (mat->getBCversion() || (!mat->getBCversion() && mat->getBCnum() == 0));
    numBC = mat->getBCnum();
    homConstraints = mat->getBCversion() || (numBC == 0);
    
    if (oneD) {
        factorize1i();
        return;
    }
    factorizeExec();
}

GridFkt LSCM::LinLSQSolver::solve(const VectorXd& f)
{
    if (!DecompositionAvail) factorize();
    
    if (oneD) return solve1i(f);
    
    return solveExec(f);
}
