/* File: LinLSQMatrices.hpp
 *
 * This class is intended to approximate the solution to linear DAEs for a
 * given grid and approximation space. It will compute the necessary matrices
 * as needed.
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2020-04-27
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LINLSQMATRICES_HPP
#define LINLSQMATRICES_HPP

#include "LSCMConfig.hpp"
#include "DAE.hpp"
#include "Xn.hpp"
#include "Yn.hpp"
#include "Grid.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <memory>
#include <vector>

namespace LSCM {

/**
 * Main class for generating the discrete matrices for linear DAEs
 * 
 * NOTE: If an LinLSQMatrices object has been changed by caling one
 * of the setX() calls, the linear solver must be notified by that
 * by calling replaceLSQMatrices() before any call to factorize() or
 * solve()! Otherwise, an undefined behavior
 * will result which does not necessarily lead to a crash.
 */
class LinLSQMatrices {
private:
    const std::shared_ptr<DAE> dae;
    std::shared_ptr<Xn> space;
    std::shared_ptr<Yn> colloc;
    std::shared_ptr<Grid> grid;
    std::vector<bool> D;  // For easier access
    LSCMint n, m, k, l, N, M, nun; // Everywhere needed
    
    // weight for boundary conditions
    double alpha = 1e0;
    
    // define norm for minimization
    Yn::LSQnorm lsqnorm;
    
    // How to handle boundary conditions:
    //   true:  Use the proven LSCM
    //   false: Require the boundary conditions to be fulfilled exactly
    bool BCversion = true;
    
public:
    /**
     * Constructor
     * 
     * @param[in] dae description of the DAE
     * @param[in] Xn approximation space
     * @param[in] colloc collocation criteria
     */
    LinLSQMatrices(const std::shared_ptr<DAE> dae, std::shared_ptr<Xn> space,
                 std::shared_ptr<Yn> colloc);
    
                 
    /**
     * Set the weight for the boundary conditions
     * 
     * @param[in] bcweight alpha;
     */
    void setAlpha(const double bcweight) {
        if (bcweight <= 0.0) {
            std::cerr << "Error setAlpha: Positive weight required. Weight not changed" << std::endl;
        }
        else alpha = bcweight;
    }
    
    /**
     * Get the weight for the boundary conditions
     * 
     * @returns returns alpha;
     */
    double getAlpha() const { return alpha; }
    
    /**
     * Set the procedure for handling the boundary conditions
     * 
     * @param[in] BCversion If set to true, the boundary conditions are 
     *              added to the functional. If set to false, the boundary
     *              conditions are taken as equality constraints.
     *              Note that the latter version is not yet proven to
     *              converge, but their are strong indications that it
     *              does anyway.
     */
    void setBCversion(bool version) { BCversion = version; }
    
    /*
     * Returns the state of BCversion, that is the procedure for handling
     * the boundary conditions
     */
    bool getBCversion() const { return BCversion; }
    
    /**
     * Check if the DAE is really an initial value problem
     */
    bool isIVP() const {
        return !((dae->getGb()).array() != 0.0).any();
    }
    
    /**
     * Unfortunately needed: returns number of boundary conditions
     */
    LSCMint getBCnum() { return l; }

   /**
     * Utility function for the nonlinear solver
     * Assembly of the global DAE right-hand side
     */
    Eigen::VectorXd feval(const GridFkt x) const;
    
    /**
     * Assembly of the global continuity matrix
     */
    LSCMSparseMatrix genC() const;
    
    /**
     * Assembly of the global DAE matrix. The number of subintervals in the grid
     * must be larger than 1.
     */
    LSCMSparseMatrix genA() const;
    
    /**
     * Assembly of the global DAE matrix for the case of of a grid with just one
     * subinterval
     */
    Eigen::MatrixXd genA1i() const;
    
     /**
     * Assembly of the global DAE right-hand side
     */
    Eigen::VectorXd genrhsq() const;
    
   // Unfortunately, needed!
    /**
     * Returns a pointer to the approximation space. Needed in order to
     * allow the solver library to return grid functions.
     * 
     * @returns a pointer to the underlying approximation space.
     */
    std::shared_ptr<Xn> getspace() const { return space; }
    
private:

    /**
     * Computes the collocation matrix on a given subinterval
     * 
     * @param[in] j subinterval
     */
    Eigen::MatrixXd gensub(LSCMint j) const;
    
    /**
     * Computes the right-hand side at the collocation points in a subinterval
     * 
     * @param[in] j subinterval
     */
    Eigen::VectorXd gensubrhs(LSCMint j) const;
    
    /**
     * Utility function for the nonlinear solver
     * Assembly of the DAE right-hand side on a subinterval
     * 
     * @param[in] j subinterval
     */
    Eigen::VectorXd fevalsub(LSCMint j, const GridFkt x) const;
    
    /**
     * Computation of matrix for continuity conditions
     *
     * @param[in] j index of grid point
     */
    Eigen::MatrixXd gencont(LSCMint j) const;

    /**
     * Returns collocation matrix for left boundary condition
     */
    Eigen::MatrixXd genGa() const;
    
    /**
     * Returns collocation matrix for right boundary condition
     */
    Eigen::MatrixXd genGb() const;

};

}

#endif
