/* File: LegendreD.hpp
 *
 * This class implements the interface of class Basis using Legendre polynomials
 * for representing polynomials. Version 2
 * 
 * C Michael Hanke 2019
 * Version: 2020-01-30
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LEGENDRED_HPP
#define LEGENDRED_HPP

#include "LSCMConfig.hpp"
#include "Basis.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <vector>

namespace LSCM {

/**
 * Implementation of the Legendre basis for the DAE (Version 2)
 * 
 * This class implements the interface of class Basis using Legendre polynomials
 * for representing polynomials for the differential components.
 * 
 */
class LegendreD : public Basis {
private:
    Eigen::VectorXd legendreP(double tau) const;
    
public:
    /**
     * Constructor for the modified Legendre basis
     * 
     * The polynomial degree N of the differential components is
     * provided. 
     * 
     * @param[in] N degree of the approximating differential polynomial
     */
    LegendreD(LSCMint N) : Basis(N) {}
    
    Eigen::VectorXd evalPa(double tau) const override {
        Eigen::VectorXd p = evaldPd(tau);
        return p.tail(dim);
    }
    Eigen::VectorXd evalPd(double tau) const override {
        Eigen::VectorXd tmp = legendreP(tau);
        double cf = 1.0;
        for (LSCMint i = 1; i <= dim; ++i) {
            tmp(i) += cf;
            cf = -cf;
        }
        return tmp;
    }
    Eigen::VectorXd evaldPd(double tau) const override;
};

}

#endif
