/* File: Legendre.hpp
 *
 * This class implements the interface of class Basis using Legendre polynomials
 * for representing polynomials.
 * 
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-12-05
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LEGENDRE_HPP
#define LEGENDRE_HPP

#include "LSCMConfig.hpp"
#include "Basis.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <vector>
#include <cmath>

namespace LSCM {

/**
 * Implementation of the Legendre basis for the DAE
 * 
 * This class implements the interface of class Basis using Legendre polynomials
 * for representing polynomials.
 * 
 */
class Legendre : public Basis {
private:
    const bool normalize;
    
    Eigen::VectorXd legendreP(double tau, LSCMint n) const;
    
public:
    /**
     * Constructor for the Legendre basis
     * 
     * The polynomial degree N of the differential components is
     * provided. 
     * 
     * @param[in] N degree of the approximating differential polynomial
     * @param[in] normalized if true, used normalized Legendre polynomials
     */
    Legendre(LSCMint N, bool normalized = false) : Basis(N), normalize(normalized) {}
    
    Eigen::VectorXd evalPa(double tau) const override {
        Eigen::VectorXd res = legendreP(tau,dim);
        if (normalize)
            for (LSCMint i = 0; i < res.size(); ++i)
                res(i) *= std::sqrt(2.0*i+1.0);
        return res;
    }

    Eigen::VectorXd evalPd(double tau) const override;
    Eigen::VectorXd evaldPd(double tau) const override;
};

}

#endif
