/* File: Legendre.cpp
 *
 * This class implements the interface of class Basis using Legendre polynomials
 * for representing polynomials.
 * 
 * Copyright (C) Michael Hanke 2019
 * Version: 2018-08-19
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Legendre.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <cstdlib>
#include <vector>
#include <cmath>

using namespace std;
using namespace Eigen;

 VectorXd LSCM::Legendre::legendreP(double tau, LSCMint n) const {
    VectorXd res(n);
    res(0) = 1.0;
    if (n == 1) return res;
    double x = 2*tau-1.0;
    res(1) = x;
    for (LSCMint i = 2; i < n; ++i) {
        double tmp = x*res(i-1);
        res(i) = (((double) (i-1))/i)*(tmp-res(i-2))+tmp;
    }
    return res;
}

VectorXd LSCM::Legendre::evalPd(double tau) const {
    VectorXd res(dim+1);
    VectorXd p = legendreP(tau,dim+1);
    res(0) = 1.0;
    res(1) = tau;
    // This approach may be questionable!
    for (LSCMint i = 2; i <= dim; ++i)
        res(i) = (p(i)-p(i-2))/(2*(2*i-1));
    if (normalize)
        for (LSCMint i = 1; i <res.size(); ++i)
            res(i) *= sqrt(2.0*i-1.0);
    return res;
}

VectorXd LSCM::Legendre::evaldPd(double tau) const {
    VectorXd res(dim+1);
    res(0) = 0.0;
    res.segment(1,dim) = legendreP(tau,dim);
    if (normalize)
        for (LSCMint i = 1; i <res.size(); ++i)
            res(i) *= sqrt(2.0*i-1.0);
    return res;
}
