/* File: LSQSolver.hpp
 *
 * This class is intended to approximate the solution to nonlinear DAEs for a
 * given grid and approximation space.
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2022-06-10
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef LSQSOLVER_HPP
#define LSQSOLVER_HPP

#include "LSCMConfig.hpp"
#include "NlDAE.hpp"
#include "LinLSQSolver.hpp"
#include "Xn.hpp"
#include "Yn.hpp"
#include "Grid.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <functional>
#include <memory>

namespace LSCM {

/**
 *
 * This class is intended to approximate the solution to nonlinear DAEs for a
 * given grid and approximation space.
 */
class LSQSolver {
private:
    std::shared_ptr<NlDAE> dae;
    std::shared_ptr<Xn> space;
    std::shared_ptr<Yn> colloc;

    // weight for boundary conditions
    double alpha = 1e0;
    
    // define norm for minimization
    Yn::LSQnorm lsqnorm;
    
    // Select the linear solver in the iterations
    LinLSQSolver& linsolver;
    
    // Parameters for Newton iteration
    LSCMint itmax = 20;
    LSCMint imax = 5;
    double tol = 1e-6;
    double lambdamin = 1e-6;
    
    const bool BCversion = true;
    
public:
    /**
     * Constructor
     * 
     * @param[in] dae description of the nonlinear DAE
     * @param[in] space approximation space
     * @param[in] colloc collocation criteria
     * @param[in] linslv linear solver
     * @param[in] BCversion decides on how to handle the boundary
     *             conditions: either as part of the functional (true)
     *             or as equality constraint (false)
     */
    LSQSolver(std::shared_ptr<NlDAE> dae, std::shared_ptr<Xn> space,
                 std::shared_ptr<Yn> colloc, LinLSQSolver& linslv,
                 bool BCversion = true) :
                 dae(dae), space(space), colloc(colloc), linsolver(linslv), BCversion(BCversion) {}
                 
    // Create constants needed for norm evaluation
    /**
     * Set the weight for the boundary conditions
     * 
     * @param[in] bcweight alpha;
     */
    void setAlpha(const double bcweight) {
        if (bcweight <= 0.0) {
            std::cerr << "Error setAlpha: Positive weight required. Weight not changed" << std::endl;
        }
        else alpha = bcweight;
    }
    /**
     * Set maximum number of iteration for the Gauss-Newton method
     * 
     * @param[in] itmax_ maximum allowed
     */
    void setItmax(const LSCMint itmax_) {
        if (itmax_ <= 0)
            std::cerr << "Warning: itmax not changed" << std::endl;
        else
            itmax = itmax_;
    }
    
    /**
     * Set the maximum number of damping cycles when damping
     * 
     * @param[in] imax_ maximum allowed
     */
     void setImax(const LSCMint imax_) {
        if (imax_ <= 0)
            std::cerr << "Warning: imax not changed" << std::endl;
        else
            imax = imax_;
    }
    
    /**
     * Set the weight for the continuity conditions
     * 
     * @param[in] tol_ tolerance for Gauss-Newton
     */
    void setTol(const double tol_) {
        if (tol_ <= 0)
            std::cerr << "Warning: tol not changed" << std::endl;
        else
            tol = tol_;
    }

    /**
     * Solve the problem by a simple undamped Gauss-Newton method
     * 
     * @param[in] x0 initial guess
     * @returns the solution
     */
    GridFkt solve(std::function<Eigen::VectorXd(double)> x0) const;
    
    /**
     * Solve the problem by a simple undamped Gauss-Newton method with the
     * trivial function as a starting guess
     * 
     * @returns the solution
     */
    GridFkt solve() const {
        LSCMint m =(dae->getD()).size();
        auto x0 = [m](double t) { return Eigen::VectorXd::Zero(m); };
        return solve(x0);
    }
    
    /**
     * Solve the problem by a simple damped Gauss-Newton method
     * 
     * @param[in] x0 initial guess
     * @returns the solution
     */
    GridFkt solveD(std::function<Eigen::VectorXd(double)> x0) const;
    
    /**
     * Solve the problem by a simple damped Gauss-Newton method with the
     * trivial function as a starting guess
     * 
     * @returns the solution
     */
    GridFkt solveD() const {
        LSCMint m =(dae->getD()).size();
        auto x0 = [m](double t) { return Eigen::VectorXd::Zero(m); };
        return solveD(x0);
    }
    
    /**
     * Solve the problem by a sophisticated damped Gauss-Newton method developed
     * by Deuflhard et al.
     * 
     * @param[in] x0 initial guess
     * @returns the solution
     */
    GridFkt nlsq(std::function<Eigen::VectorXd(double)> x0) const;
    
    /**
     * Solve the problem by a sophisticated damped Gauss-Newton method developed
     * by Deuflhard et al. Starting guess is the zero function
     * 
     * @returns the solution
     */
    GridFkt nlsq() const {
        LSCMint m =(dae->getD()).size();
        auto x0 = [m](double t) { return Eigen::VectorXd::Zero(m); };
        return nlsq(x0);
    }
};

}

#endif
