/* File: GridFkt.hpp
 * 
 * This file implements the handling of grid functions.
 * 
 * Taking into account future applications including quasilinearization and
 * grid adaptation, we should be able to manipulate functions living on
 * different grids and/or function spaces.
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2019-12-03
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef GRIDFKT_HPP
#define GRIDFKT_HPP

#include "LSCMConfig.hpp"
#include "Xn.hpp"
#include "Grid.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <functional>
#include <memory>
#include <iostream>
#include <cstdlib>

namespace LSCM {

/**
 * This class implements the handling of grid functions.
 * 
 * This class contains all necessary constructors and operation for handling
 * grid function living on different grids and spaces. Thus, the necessary
 * projections are defined. Both L2 and H1 projections are available. As a default,
 * the H1 projection is used.
 * 
 * It should be noted that, in arithmetic operations, the final result lives
 * in the space of the left-hand operand! So it is a good idea to have the
 * finer space attached to the left-hand operand.
 */
class GridFkt {
public:
    /**
     * This type is used for selecting the projection method for eliminating jumps
     * in the discrete solution.
     */
    typedef enum { JMP_SPQR,   /**< project by computing a nullspace basis */
                   JMP_LU,     /**< use LU-decomposition for calculating the orthoprojector */
                   JMP_LAPACK  /**< use the LAPACK tridiagonal solver */
    } JMP_TYPE;
private:
    std::shared_ptr<Xn> space = nullptr;
    Eigen::VectorXd coeffs;
    std::vector<bool>& D;   // TODO: Better solution: This version does not allow for the
                            // default constructor!
                            // Seems a conceptual bug: Instead of only D, the space must be
                            // tied with the complete DAE! Then this become a shared_ptr<DAE>!
                            // This in turn requires a unique interface for both linear and
                            // nonlinear DAEs.
    std::shared_ptr<Grid> grid = nullptr;
    LSCMint alldofs = 0;
    LSCMint n = 0, m = 0, k = 0, N = 0, nun = 0;
    
    bool isSpaceIdentical(const GridFkt& g) const {
        return space == g.space;
    }
    
    void check_consistency(const GridFkt& g) const {
        if (m != g.getm()) {
            std::cerr << "GridFkt: Incompatible grid functions" << std::endl;
            exit(1);
        }
        double a = geta();
        double b = getb();
        //std::cerr << "consistency: a = " << a << ", b = " << b << std::endl;
        double tol = (b-a)*std::numeric_limits<double>::epsilon()*10.0;
        if ((abs(a-g.geta()) > tol) || (abs(b-g.getb()) > tol)) {
            std::cerr << "GridFkt: Incompatible grid functions" << std::endl;
            exit(1);
        }
        for (LSCMint i = 0; i < m; ++i) {
           if (D[i] != g.D[i]) {
                std::cerr << "GridFkt: Incompatible grid functions" << std::endl;
                exit(1);
           }
        }
    }
    
    Eigen::MatrixXd gencont(LSCMint nn) const;
    std::vector<LSCMTriplet> gensub(LSCMint j) const;
    std::vector<LSCMTriplet> gensubd(LSCMint j) const;
    Eigen::VectorXd dirclsq(const LSCMSparseMatrix& A,
                            const LSCMSparseMatrix& C,
                            const Eigen::VectorXd& rhs) const;
                                    
    Eigen::VectorXd L2interpol(std::function<Eigen::VectorXd(const double)>) const;
    Eigen::VectorXd H1interpol(std::function<Eigen::VectorXd(const double)>,
        std::function<Eigen::VectorXd(const double)>) const;
    
public:
    // BUG: This constructor is deleted! Find a nice way for the default constructor!
    /**
     * Default constructor
     */
    GridFkt() = default;
    
    /**
     * Constructor: For internal use
     * 
     * @param[in] space ansatz space
     * @param[in] coeffs vector of coefficients: Respect the order defined by space!
     * if no coefficient vector is given, the zero function is constructed.
     */
    GridFkt(std::shared_ptr<Xn> space,
        const Eigen::VectorXd& coeffs = Eigen::VectorXd(0));
    
    /**
     * Constructor: For internal use
     * 
     * @param[in] space ansatz space
     * @param[in] coeffs vector of coefficients: Respect the order defined by space!
     */
    GridFkt(std::shared_ptr<Xn> space, Eigen::VectorXd&& coeffs);
    
    /**
     * Copy constructor
     * 
     * @param[in] g a GridFunction object
     */
    GridFkt(const GridFkt& g) : space(g.space), coeffs(g.coeffs),
        D(g.D), grid(g.grid), alldofs(g.alldofs), n(g.n), m(g.m), k(g.k), N(g.N),
        nun(g.nun) {}
            
   /**
     * Constructor using the L2 projection
     * 
     * @param[in] space ansatz space
     * @param[in] fkt function for intializing
     */
    GridFkt(std::shared_ptr<Xn> space,
        std::function<Eigen::VectorXd(double)> fkt);
    
     /**
     * Constructor using the H1 projection
     * 
     * @param[in] space ansatz space
     * @param[in] fkt function for initializing
     * @param[in] dfkt the derivative of fkt
     */
    GridFkt(std::shared_ptr<Xn> space,
        std::function<Eigen::VectorXd(double)> fkt,
        std::function<Eigen::VectorXd(double)> dfkt);

    /**
     * Constructor with possibly H1 projection
     * 
     * @param[in] space ansatz space
     * @param[in] g     Grid function that will be projected onto space
     */
    GridFkt(std::shared_ptr<Xn> space,
        const GridFkt& g);
    
    /**
     * Projection of the grid function onto H^1_D
     * 
     * This projection realizes the projection of the coefficients in R^n. The underlying
     * space will not be changed. Its intended use is the projection of a perturbed
     * grid function (e.g., the result of a solve()) to an element of the approximation
     * space. For that, the continuity of the differentiated components will be ensured.
     * 
     * @param[in] version selects the projector version to be used.
     */
    void project(const JMP_TYPE version = JMP_SPQR);
    
    /**
     * Computation of the jump at the gridpoints
     * 
     * The jump of the grid function can be evaluated at all internal grid points. It should
     * be zero for the differentiated components.
     * 
     * @param[in] j index of gridpoint. It must hold 0 < j < n (n - number of subintervals)
     */
    Eigen::VectorXd jump(LSCMint j) const;
    
    // Standard operators
    /**
     * Copy assignment operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt& operator=(const GridFkt& g);
    
    /**
     * Addition operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt operator+(const GridFkt& g) const;
    
    /**
     * Subtraction operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt operator-(const GridFkt& g) const;
    
    /**
     * Division operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt operator/(const GridFkt& g) const;
    
    /**
     * Multiplication operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt operator*(const GridFkt& g) const;
    
    /**
     * Addition-assignment operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt& operator+=(const GridFkt& g);
    
    /**
     * Multiplication-assignment operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt& operator*=(const GridFkt& g);
    
    /**
     * Division-assignment operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt& operator/=(const GridFkt& g);
    
    /**
     * Subtraction-assignment operator
     * 
     * @param[in] g GridFkt object
     */
    GridFkt& operator-=(const GridFkt& g);
    
    /**
     * Multiplcation by a scalar operator
     * 
     * @param[in] al scalar
     */
    GridFkt operator*(const double al) const;
    
    /**
     * Multiplcation by a scalar operator
     * 
     * @param[in] al scalar
     * @param[in] g GridFkt object
     * @returns product
     */
    friend GridFkt operator*(const double al, const GridFkt& g);
    
    /**
     * Scaling by a vector
     * 
     * @param[in] scal vector of scaling coeficients
     * @returns scaled grid function
     */
    GridFkt operator*(const Eigen::VectorXd scal) const;
    
    /**
     * Scaling by a vector
     * 
     * @param[in] invscal vector of scaling coeficients
     * @returns scaled grid function
     */
    GridFkt operator/(const Eigen::VectorXd invscal) const
    {
        Eigen::VectorXd scal(invscal.size());
        for (LSCMint i = 0; i < scal.size(); ++i) scal(i) = 1.0/invscal(i);
        return (*this)*scal;
    }
    
    // And function vales must be able to be computed!
    /**
     * Evaluation of function values
     * 
     * @param[in] t evaluation point
     * @returns value of the function
     */
    Eigen::VectorXd eval(const double t) const;
    
    /**
     * Evaluation of function values
     * 
     * @param[in] t evaluation point
     * @returns value of the function
     */
    Eigen::VectorXd operator()(const double t) const
    {
        return eval(t);
    }
    
    /**
     * Evaluation of the derivative of the function
     * 
     * Note that the derivative of algebraic components is set to zero
     * 
     * @param[in] t evaluation point
     * @returns derivative of the function
     */
    // The derivative of non-differential components is set to zero!
    Eigen::VectorXd deval(const double t) const;
    
    std::shared_ptr<Grid> getGrid() { return grid; }
    
    std::shared_ptr<Xn> getSpace() { return space; }
    
    const Eigen::VectorXd& getcoeffs() const { return coeffs; }
    
    /**
     * Returns the left interval boundary
     * 
     * @returns the left interval boundar
     */
    double geta() const { return (*grid)[0]; }
    
    /**
     * Returns the right interval boundary
     * 
     * @returns the right interval boundary
     */
    double getb() const { return (*grid)[n]; }
    
    /**
     * Returns the dimension of the function
     * 
     * @returns the dimension of the function
     * 
     */
    LSCMint getm() const { return space->getm(); }
    
    /**
     * Computes the L2-norm
     * 
     * @returns the L2-norm
     */
    double l2norm() const;
    
    /**
     * Computes the H1D-norm
     * 
     * @returns the H1D-norm
     */
    double h1norm() const;
    
    /**
     * Computes the maximum norm
     * 
     * @returns the maximum norm
     */
    double infnorm() const;
    
    /**
     * Computes the componentwise L2-norm
     * 
     * @returns the componentwise L2-nor
     */
    Eigen::VectorXd cl2norm() const;
    
    /**
     * Computes the componentwise maximum norm
     * 
     * @returns the componentwise maximum norm
     */
    Eigen::VectorXd cinfnorm()const ;
    
    /**
     * Computes the L2-distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @returns the L2-distance
     */
    double l2dist(std::function<Eigen::VectorXd(double)> exsol) const;
    
    /**
     * Computes the L2-distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @returns the L2-distance
     */
    double l2dist(Eigen::VectorXd exsol(const double t)) const
    {
        return l2dist(std::bind(exsol, std::placeholders::_1));
    }
    
    /**
     * Computes the maximum distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @returns the maximum distance
     */
    double infdist(std::function<Eigen::VectorXd(double)> exsol) const;
    
    /**
     * Computes the maximum distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @returns the maximum distance
     */
    double infdist(Eigen::VectorXd exsol(const double t)) const
    {
        return infdist(std::bind(exsol, std::placeholders::_1));
    }
    
    /**
     * Computes the H1D-distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @param[in] dexsol derivative of the differential components of exsol
     * @returns the H1D-distance
     */
    double h1dist(std::function<Eigen::VectorXd(double)> exsol,
                  std::function<Eigen::VectorXd(double)> dexsol) const;
    
    /**
     * Computes the H1D-distance of the grid function from an analytically given one.
     * 
     * @param[in] exsol function
     * @param[in] dexsol derivative of the differential components of exsol
     * @returns the H1D-distance
     */
    double h1dist(Eigen::VectorXd exsol(const double t),
                  Eigen::VectorXd dexsol(const double t)) const 
    {
        return h1dist(std::bind(exsol, std::placeholders::_1),
            std::bind(dexsol, std::placeholders::_1));
    }
    
    /**
     * Computes the componentwise L2-distance of the grid function from an analytically 
     * given one.
     * 
     * @param[in] exsol function
     * @returns the componentwise L2-distance
     */
    Eigen::VectorXd cl2dist(std::function<Eigen::VectorXd(double)> exsol) const;
    
    /**
     * Computes the componentwise L2-distance of the grid function from an analytically
     * given one.
     * 
     * @param[in] exsol function
     * @returns the componentwise L2-distance
     */
    Eigen::VectorXd cl2dist(Eigen::VectorXd exsol(const double t)) const
    {
        return cl2dist(std::bind(exsol, std::placeholders::_1));
    }
    
    /**
     * Computes the componentwise maximum distance of the grid function from an
     * analytically given one.
     * 
     * @param[in] exsol function
     * @returns the componentwise maximum distance
     */
    Eigen::VectorXd cinfdist(std::function<Eigen::VectorXd(double)> exsol) const;
    
    /**
     * Computes the componentwise maximum distance of the grid function from an
     * analytically given one.
     * 
     * @param[in] exsol function
     * @returns the componentwise maximum distance
     */
    Eigen::VectorXd cinfdist(Eigen::VectorXd exsol(const double t)) const
    {
        return cinfdist(std::bind(exsol, std::placeholders::_1));
    }
    
    /**
     * Helper function for error estimation. It returns the H1-norm of the function
     * on each subinterval.
     * 
     * @returns the vector of local norms
     */
    Eigen::VectorXd h1locnorm() const;
    
    /**
     * Helper function for error estimation. It returns the L2-norm of the function
     * on each subinterval.
     * 
     * @returns the vector of local norms
     */
    Eigen::VectorXd l2locnorm() const;
    
    /**
     * Helper function for error estimation. It returns the C-norm of the function
     * on each subinterval.
     * 
     * @returns the vector of local norms
     */
    Eigen::VectorXd inflocnorm() const;
    
    /*
     * Helper for testing: Creates a function being 0 and 1 on adjacent
     * subintervals
     */
    //GridFkt mismatch() const;
};

/**
 * Multiplication operator for doubles and grid functions
 */
GridFkt operator*(const double al, const GridFkt& g);

}

#endif
