/* File: GridFkt.cpp
 * 
 * This file implements the handling of grid functions.
 * 
 * Taking into account future applications including quasilinearization and
 * grid adaptation, we should be able to manipulate functions living on
 * different grids and/or function spaces.
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2018-08-20
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "GridFkt.hpp"
#include "QuadratureRule.hpp"
#include "Xn.hpp"
#include "Grid.hpp"
#include <Eigen/Dense>
#include <Eigen/SparseCore>
#include <Eigen/SPQRSupport>
#include <Eigen/SparseLU>
#include <functional>
#include <memory>
#include <iostream>
#include <cstdlib>
#include <cmath>

using namespace LSCM;
using namespace std;
using namespace Eigen;

LSCM::GridFkt::GridFkt(shared_ptr<Xn> space_, const VectorXd& coeffs_ ) :
space(space_), D(space_->getDptr()), grid(space_->getgrid()),
n((space_->getgrid())->getn()), m(space_->getm()), k(space_->getk()),
N(space_->getN()), nun(space_->getdim())
{
    // Some checks and initializations
    alldofs = n*space->getdim();
    
    if (coeffs_.size() == 0) {
        // OK, zero function
        coeffs = VectorXd::Zero(alldofs);
    }
    else {
        if (coeffs_.size() == alldofs) {
            coeffs = coeffs_;
        }
        else {
            cerr << "Error GridFkt: Invalid size of coefficient vector" << endl;
            cerr << "Expected: " << alldofs << ", actual size: " << coeffs_.size() << endl;
            exit(1);
        }
    }
}

LSCM::GridFkt::GridFkt(shared_ptr<Xn> space_, VectorXd&& coeffs_ ) :
space(space_), D(space_->getDptr()), grid(space_->getgrid()),
n((space_->getgrid())->getn()), m(space_->getm()), k(space_->getk()),
N(space_->getN()), nun(space_->getdim())
{
    // Some checks and initializations
    alldofs = n*space->getdim();
    
    if (coeffs_.size() == alldofs) {
        coeffs = std::move(coeffs_);
    }
    else {
        cerr << "Error GridFkt: Invalid size of coefficient vector" << endl;
        cerr << "Expected: " << alldofs << ", actual size: " << coeffs_.size() << endl;
        exit(1);
    }
}

LSCM::GridFkt::GridFkt(shared_ptr<Xn> space_, const GridFkt& g) :
space(space_), D(space_->getDptr()), grid(space_->getgrid()),
n((space_->getgrid())->getn()), m(space_->getm()), k(space_->getk()),
N(space_->getN()), nun(space_->getdim())
{
    // Check dimensions
    check_consistency(g);
    
    // initialize
    alldofs = n*space->getdim();
    
    function<VectorXd(const double)> wrapper =
    [&g](const double t) -> VectorXd { return g.eval(t); };
    function<VectorXd(const double)> dwrapper =
    [&g](const double t) -> VectorXd { return g.deval(t); };
    coeffs = H1interpol(wrapper,dwrapper);
}

LSCM::GridFkt::GridFkt(shared_ptr<Xn> space_, function<VectorXd(double)> fkt) :
space(space_), D(space_->getDptr()), grid(space_->getgrid()),
n((space_->getgrid())->getn()), m(space_->getm()), k(space_->getk()),
N(space_->getN()), nun(space_->getdim())
{
    VectorXd res = fkt((*grid)[0]);
    if (m != res.size()) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
        }
    
    alldofs = n*space->getdim();
    coeffs = L2interpol(fkt);
}

LSCM::GridFkt::GridFkt(shared_ptr<Xn> space_, function<VectorXd(double)> fkt,
    function<VectorXd(double)> dfkt) :
space(space_), D(space_->getDptr()), grid(space_->getgrid()),
n((space_->getgrid())->getn()), m(space_->getm()), k(space_->getk()), N(space_->getN()),
nun(space_->getdim())
{
    VectorXd res = fkt((*grid)[0]);
    VectorXd dres = dfkt((*grid)[0]);
    if ((m != res.size()) || (m != dres.size())) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
    }
    
    alldofs = n*space->getdim();
    coeffs = H1interpol(fkt,dfkt);
}

VectorXd LSCM::GridFkt::L2interpol(function<VectorXd(double)> fkt) const {
    // Find some constants
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    LSCMint rdim = n*m*M;
    LSCMint nnz = M*alldofs;
    // Dimensions CM
    LSCMint nnzc = 2*k*nun*(n-1);
    LSCMint rdimc = (n-1)*k;
    
    vector<LSCMTriplet> DOK;
    DOK.reserve(nnz);   // How much??
    vector<LSCMTriplet> DOKc;
    DOKc.reserve(nnzc);   // How much??
    
    LSCMint iglob = 0;
    LSCMint jglob = 0;
    for (LSCMint nn = 1; nn <= n; ++nn) {
        vector<LSCMTriplet> AB = gensub(nn-1);
        for (LSCMint l = 0; l < AB.size(); ++l)
            DOK.push_back(LSCMTriplet(AB[l].row()+iglob,
                                          AB[l].col()+jglob,
                                          AB[l].value()));
        iglob += m*M;
        jglob += nun;
    }
    
    iglob = 0;
    jglob = 0;
    for (LSCMint nn = 1; nn < n; ++nn) {
        MatrixXd CM = gencont(nn);
        for (LSCMint j = 0; j < 2*nun; ++j)
            for (LSCMint i = 0; i < k; ++i)
                DOKc.push_back(LSCMTriplet(iglob+i,jglob+j,CM(i,j)));
        iglob += k;
        jglob += nun;
    }

    // Create sparse matrices
    LSCMSparseMatrix C(rdimc,alldofs);
    C.setFromTriplets(DOKc.begin(),DOKc.end());
    struct keepFkt {
        bool operator() (const Index& row, const Index& col, const double& value) const {
            return value != 0.0;
        }
    } kF;
    C.prune(kF);
    //C.makeCompressed();
    LSCMSparseMatrix A(rdim,alldofs);
    A.setFromTriplets(DOK.begin(),DOK.end());
    A.prune(kF);          // Very expensive!!!
    //A.makeCompressed();

    // Compute the right-hand side
    VectorXd rhs(rdim);
    LSCMint ioff = 0;
    for (LSCMint nn = 1; nn <= n; ++nn) {
        double tj1 = (*grid)[nn-1];
        double h = (*grid)[nn]-tj1;
        for (LSCMint l = 0; l < M; ++l) {
            VectorXd Fl = sqrt(sw[l].weight*h)*fkt(h*sw[l].zero+tj1);
            rhs.segment(ioff,m) = Fl;
            ioff += m;
        }
    }

    // Finally, solve! We use the direct solver
    return dirclsq(A,C,rhs);
}

VectorXd LSCM::GridFkt::H1interpol(function<VectorXd(double)> fkt,
    function<VectorXd(double)> dfkt) const {
    // Find some constants
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    LSCMint rdim = n*(m+k)*M;
    LSCMint nnz = 2*n*M*nun;
    // Dimensions CM
    LSCMint nnzc = 2*k*nun*(n-1);
    LSCMint rdimc = (n-1)*k;
    
    vector<LSCMTriplet> DOK;
    DOK.reserve(nnz);   // How much??
    vector<LSCMTriplet> DOKc;
    DOKc.reserve(nnzc);   // How much??
    
    LSCMint iglob = 0;
    LSCMint jglob = 0;
    for (LSCMint nn = 1; nn <= n; ++nn) {
        vector<LSCMTriplet> AB = gensub(nn-1);
        for (LSCMint l = 0; l < AB.size(); ++l)
            DOK.push_back(LSCMTriplet(AB[l].row()+iglob,
                                          AB[l].col()+jglob,
                                          AB[l].value()));
        iglob += m*M;
        AB = gensubd(nn-1);
        for (LSCMint l = 0; l < AB.size(); ++l)
            DOK.push_back(LSCMTriplet(AB[l].row()+iglob,
                                          AB[l].col()+jglob,
                                          AB[l].value()));
        iglob += k*M;
        jglob += nun;
    }

    iglob = 0;
    jglob = 0;
    for (LSCMint nn = 1; nn < n; ++nn) {
        MatrixXd CM = gencont(nn);
        for (LSCMint j = 0; j < 2*nun; ++j)
            for (LSCMint i = 0; i < k; ++i)
                DOKc.push_back(LSCMTriplet(iglob+i,jglob+j,CM(i,j)));
        iglob += k;
        jglob += nun;
    }

    // Create sparse matrices
    LSCMSparseMatrix C(rdimc,alldofs);
    C.setFromTriplets(DOKc.begin(),DOKc.end());
    struct keepFkt {
        bool operator() (const Index& row, const Index& col, const double& value) const {
            return value != 0.0;
        }
    } kF;
    C.prune(kF);
    //C.makeCompressed();
    LSCMSparseMatrix A(rdim,alldofs);
    A.setFromTriplets(DOK.begin(),DOK.end());
    A.prune(kF);
    //A.makeCompressed();

    // Compute the right-hand side
    VectorXd rhs(rdim);
    LSCMint ioff = 0;
    for (LSCMint nn = 1; nn <= n; ++nn) {
        double tj1 = (*grid)[nn-1];
        double h = (*grid)[nn]-tj1;
        for (LSCMint l = 0; l < M; ++l) {
            VectorXd Fl = sqrt(sw[l].weight*h)*fkt(h*sw[l].zero+tj1);
            rhs.segment(ioff,m) = Fl;
            ioff += m;
        }
        for (LSCMint l = 0; l < M; ++l) {
            VectorXd Fld = sqrt(sw[l].weight*h)*dfkt(h*sw[l].zero+tj1);
            for (LSCMint i = 0; i < m; ++i) {
                if (D[i]) {
                    rhs(ioff) = Fld(i);
                    ++ioff;
                }
            }   
        }
    }

    // Finally, solve! We use the direct solver
    return dirclsq(A,C,rhs);
}

MatrixXd LSCM::GridFkt::gencont(LSCMint j) const {
    LSCMint kk = k;
    
    MatrixXd ccj = MatrixXd::Zero(kk,2*nun);

    double hr = (*grid)[j+1]-(*grid)[j];
    double hl = (*grid)[j]-(*grid)[j-1];
    VectorXd psir = space->evalr(j);
    VectorXd psil = space->evall(j);

    LSCMint joff = 0;
    LSCMint koff = 0;
    for (LSCMint k = 0; k < m; k++) {
        if (D[k]) {
            for (LSCMint l = 0; l <= N; l++) {
                ccj(koff,joff+l) = psil(joff+l);
            }
            for (LSCMint l = 0; l <= N; l++) {
                ccj(koff,nun+joff+l) = -psir(joff+l);
            }
            joff += N+1;
            ++koff;
        }
        else {
            joff += N;
        }
    }
    return ccj;
}

vector<LSCMTriplet> LSCM::GridFkt::gensub(LSCMint j) const {
    LSCMint kk = k;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    
    vector<LSCMTriplet> DOK;
    DOK.reserve(M*(space->getdim()));
    
    double tj = (*grid)[j];
    double h = (*grid)[j+1]-tj;
    LSCMint koff = 0;

    for (LSCMint l = 0; l < M; ++l) {
        VectorXd psil = sqrt(sw[l].weight*h)*(space->evalsub(sw[l].zero,h));
        for (LSCMint k = 0; k < m; ++k) {
            LSCMint joff = 0;
            for (LSCMint i = 0; i < m; ++i) {
                if (D[i]) {
                    if (k == i) {
                        for (LSCMint j = 0; j <= N; ++j) {
                            DOK.push_back(LSCMTriplet(koff+k,joff+j,psil(joff+j)));
                        }
                    }
                    joff += N+1;
                }
                else {
                    if (k == i) {
                        for (LSCMint j = 0; j < N; ++j) {
                            DOK.push_back(LSCMTriplet(koff+k,joff+j,psil(joff+j)));
                        }
                    }
                    joff += N;
                }
            }
        }
        koff += m;
    }

    return DOK;
}

vector<LSCMTriplet> LSCM::GridFkt::gensubd(LSCMint j) const {
    LSCMint kk = k;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    
    vector<LSCMTriplet> DOK;
    DOK.reserve(M*(space->getdim())); // Too many!
    
    double tj = (*grid)[j];
    double h = (*grid)[j+1]-tj;
    LSCMint koff = 0;

    for (LSCMint l = 0; l < M; ++l) {
        VectorXd psil = sqrt(sw[l].weight*h)*(space->devalsub(sw[l].zero,h));
        for (LSCMint k = 0; k < m; ++k) {
            LSCMint joff = 0;
            LSCMint ioff = 0;
            for (LSCMint i = 0; i < m; ++i) {
                if (D[i]) {
                    if (k == i) {
                    for (LSCMint j = 0; j <= N; ++j) {
                            DOK.push_back(LSCMTriplet(koff+ioff,joff+j,psil(joff+j)));
                        }
                    }
                    joff += N+1;
                    ++ioff;
                }
                else joff += N;
            }
        }
        koff += kk;
    }

    return DOK;
}

VectorXd LSCM::GridFkt::dirclsq(const LSCMSparseMatrix& A,
                const LSCMSparseMatrix& C, const VectorXd& rhs) const {
    LSCMint Crow = C.rows();
    LSCMint nun = C.cols();
    if (Crow == 0) {
        SPQR<LSCMSparseMatrix> qrA;
        qrA.compute(A);
        if (qrA.info() != Success) {
            // cerr << QR.lastErrorMessage() << endl;
            cerr << "GridFkt: QR decomposition failed" << endl;
            exit(1);
        }
        VectorXd coeffs = qrA.solve(rhs);
        if (qrA.info() != Success) {
            cerr << "GridFkt: QR solve failed" << endl;
            exit(1);
        }
        return coeffs;
    }
    
    // The algorithm requires too much memory in this implementation!
    // TODO: Q is essentially the identity for this kind of C!!!! Use it!
    // This version of SPQR is not documented!!
    //SPQR<LSCMSparseMatrix> qrC(C);
    SPQR<LSCMSparseMatrix> qrC;
    qrC.compute(C);
    if (qrC.info() != Success) {
        // cerr << QR.lastErrorMessage() << endl;
        cerr << "GridFkt: QR decomposition failed" << endl;
        exit(1);
    }
    LSCMSparseMatrix U = qrC.matrixR();
    LSCMSparseMatrix U1 = U.leftCols(Crow);
    // Check fullrank
    for (LSCMint i = 0; i < Crow; ++i)
        if (U1.coeff(i,i) == 0.0) {
            cerr << "GridFkt: Continuity conditions dependent!" << endl;
            exit(1);
        }

    LSCMSparseMatrix U2 = U.rightCols(nun-Crow);
    auto P = qrC.colsPermutation();
    LSCMSparseMatrix Xperm = A*P;
    LSCMSparseMatrix X1 = Xperm.leftCols(Crow);
    LSCMSparseMatrix X2 = Xperm.rightCols(nun-Crow);
    
    // Check if U1 is diagonal
    if (U1.nonZeros() == Crow) {
        // U1 is diagonal
        // Compute the inverses in order to avoid division
        double *U1values = U1.valuePtr();
        for (LSCMint i = 0; i < Crow; ++i) U1values[i] = 1.0/U1values[i];
        // Sparse Matrix format is column major!
        for (LSCMint k = 0; k < U2.outerSize(); ++k) 
            for (LSCMSparseMatrix::InnerIterator it(U2,k); it; ++it)
                it.valueRef() *= U1.coeff(it.row(),it.row());
    }
    else {
        cerr << "GridFkt: U1 is not diagonal!" << endl;
        SparseLU<LSCMSparseMatrix,COLAMDOrdering<LSCMindex>> lu;
        lu.compute(U1.triangularView<Upper>());
        if (lu.info() != Success) {
            // cerr << QR.lastErrorMessage() << endl;
            cerr << "GridFkt: LU decomposition failed" << endl;
            exit(1);
        }
        LSCMSparseMatrix tmp = lu.solve(U2);
        if (lu.info() != Success) {
            cerr << "GridFkt: LU solve during factorization failed" << endl;
            exit(1);
        }
        U2 = tmp;
    }
    
    // This does not work!
    // SparseMatrix<double> tmp = U1.triangularView<Upper>().solve(U2);
    LSCMSparseMatrix Xtilde = X2-X1*U2;
    SPQR<LSCMSparseMatrix> qrX;
    qrX.compute(Xtilde);
    if (qrX.info() != Success) {
        // cerr << QR.lastErrorMessage() << endl;
        cerr << "GridFkt: QR decomposition failed" << endl;
        exit(1);
    }
    
    VectorXd al = qrX.solve(rhs);
    if (qrX.info() != Success) {
        // cerr << QR.lastErrorMessage() << endl;
        cerr << "GridFkt: solve failed" << endl;
        exit(1);
    }

    VectorXd tmpCoeffs(nun);
    tmpCoeffs.head(U2.rows()) = -U2*al;
    tmpCoeffs.tail(al.size()) = al;
    VectorXd coeffs = P*tmpCoeffs;
    return coeffs;
}

void LSCM::GridFkt::project(const JMP_TYPE version) {
    // Create C_s, create and decompose C_sC_s^T. This is done via local routines
    
    // First, the version via explicit computation of the Moore-Penrose inverse.
    // The linear system is solved via eigen's SparseLU.
    // Note that no memory is used if this version will not be used!
    typedef struct {
    private:
        LSCMSparseMatrix Cs;
        SparseLU<LSCMSparseMatrix,COLAMDOrdering<LSCMindex>> solver;
    public:
        void Csgen(shared_ptr<Xn> space) {
            auto f = space->getBasis()->evalPd(1.0);
            LSCMint nm1 = (space->getgrid())->getn()-1;

            // Create Cs
            LSCMint N1 = f.size();
            LSCMint nnzest = nm1*(N1+1);
            vector<LSCMTriplet> DOK;
            DOK.reserve(nnzest);
            double told = (*(space->getgrid()))[1];
            double h = told-(*(space->getgrid()))[0];
            LSCMint joff = 0;
            for (LSCMint j = 0; j < nm1; ++j) {
                for (LSCMint i = 0; i < N1; ++i)
                    DOK.push_back(LSCMTriplet(j,joff+i,h*f(i)));
                joff += N1;
                double tn = (*(space->getgrid()))[j+2];
                double hn = tn-told;
                DOK.push_back(LSCMTriplet(j,joff,-hn));
                told = tn;
                h = hn;
            }
            Cs = LSCMSparseMatrix(nm1,(nm1+1)*N1);
            Cs.setFromTriplets(DOK.begin(),DOK.end());
            Cs.prune(0.0);
            
            // Decompose Cs*CsT
            solver.compute(Cs*Cs.transpose());
        }
        VectorXd Cssolve(const VectorXd& ckappa) const {
            return VectorXd(solver.solve(ckappa));
        }
        const LSCMSparseMatrix& Csget() const { return Cs; }
    } Csdata;
    
    // This is the computation of the Moore-Penrose inverse using LAPACK
    typedef struct {
    private:
        LSCMSparseMatrix Cs;
        double *b = nullptr;
        double *d = nullptr;
        double *bf = nullptr;
        double *df = nullptr;
        LSCMint nm1;
    public:
        void LAgen(shared_ptr<Xn> space) {
            auto f = space->getBasis()->evalPd(1.0);
            nm1 = (space->getgrid())->getn()-1;
            delete [] b;
            delete [] bf;
            delete [] d;
            delete [] df;
            b = new double[nm1-1];
            bf = new double[nm1-1];
            d = new double[nm1];
            df = new double[nm1];
            
            // Create Cs
            LSCMint N1 = f.size();
            LSCMint nnzest = nm1*(N1+1);
            vector<LSCMTriplet> DOK;
            DOK.reserve(nnzest);
            double told = (*(space->getgrid()))[1];
            double h = told-(*(space->getgrid()))[0];
            LSCMint joff = 0;
            for (LSCMint j = 0; j < nm1; ++j) {
                for (LSCMint i = 0; i < N1; ++i)
                    DOK.push_back(LSCMTriplet(j,joff+i,h*f(i)));
                joff += N1;
                double tn = (*(space->getgrid()))[j+2];
                double hn = tn-told;
                DOK.push_back(LSCMTriplet(j,joff,-hn));
                told = tn;
                h = hn;
            }
            Cs = LSCMSparseMatrix(nm1,(nm1+1)*N1);
            Cs.setFromTriplets(DOK.begin(),DOK.end());
            Cs.prune(0.0);

            // Create Cs*Cs.transpose()
            double f2 = f.squaredNorm();
            told = (*(space->getgrid()))[0];
            double tm1 = (*(space->getgrid()))[1];
            double h2 = (tm1-told)*(tm1-told);
            d[0] = h2*f2;
            for (LSCMint j = 1; j < nm1; ++j) {
                told = tm1;
                tm1 = (*(space->getgrid()))[j+1];
                h2 = (tm1-told)*(tm1-told);
                b[j-1] = -h2;
                d[j-1] += h2;
                d[j] = h2*f2;
            }
            told = tm1;
            tm1 = (*(space->getgrid()))[nm1+1];
            h2 = (tm1-told)*(tm1-told);
            d[nm1-1] += h2;
            // Decompose
            double rcond,ferr,berr;
            LSCMint info = LAPACKE_dptsvx(LAPACK_COL_MAJOR,'N',nm1,0,d,b,df,bf,nullptr,nm1,
                                      nullptr,nm1,&rcond,&ferr,&berr);
            if (info > 0) {
                cerr << "GridFkt jump: numerically singular system" << endl;
            }
        }
        VectorXd LAsolve(const VectorXd& ckappa) const {
            VectorXd x(ckappa.size());
            double rcond,ferr,berr;
            LSCMint info = LAPACKE_dptsvx(LAPACK_COL_MAJOR,'F',nm1,1,d,b,df,bf,ckappa.data(),nm1,
                                      x.data(),nm1,&rcond,&ferr,&berr);
            return x;
        }
        const LSCMSparseMatrix& Csget() const { return Cs; }
        void LAfree() {
            delete [] b;
            delete [] bf;
            delete [] d;
            delete [] df;
        }
    } LAdata;
    
    // And now the version via nullspace base
    typedef struct {
    private:
        SPQR<LSCMSparseMatrix> qrCs;
        LSCMint nm1;
    public:
        void Q2gen(shared_ptr<Xn> space) {
            auto f = space->getBasis()->evalPd(1.0);
            nm1 = (space->getgrid())->getn()-1;

            // Create Cs
            LSCMint N1 = f.size();
            LSCMint nnzest = nm1*(N1+1);
            vector<LSCMTriplet> DOK;
            DOK.reserve(nnzest);
            double told = (*(space->getgrid()))[1];
            double h = told-(*(space->getgrid()))[0];
            LSCMint joff = 0;
            for (LSCMint j = 0; j < nm1; ++j) {
                for (LSCMint i = 0; i < N1; ++i)
                    DOK.push_back(LSCMTriplet(joff+i,j,h*f(i)));
                joff += N1;
                double tn = (*(space->getgrid()))[j+2];
                double hn = tn-told;
                DOK.push_back(LSCMTriplet(joff,j,-hn));
                told = tn;
                h = hn;
            }
            LSCMSparseMatrix Cs = LSCMSparseMatrix((nm1+1)*N1,nm1);
            Cs.setFromTriplets(DOK.begin(),DOK.end());
            Cs.prune(0.0);
            
            // Create Q2
            qrCs.compute(Cs);
            if (qrCs.info() != Success) {
                // cerr << QR.lastErrorMessage() << endl;
                cerr << "GridFkt: QR decomposition failed in project" << endl;
                exit(1);
            }
        }
        VectorXd Q2solve(const VectorXd& ckappa) const {
            VectorXd tmp = qrCs.matrixQ().transpose()*ckappa;
            tmp.head(nm1) = VectorXd::Zero(nm1);
            return qrCs.matrixQ()*tmp;
        }
    } Q2data;
    
    // The algorithm starts here
    Q2data Q2;
    Csdata Cs;
    LAdata LA;
    switch (version) {
        case JMP_SPQR:
            Q2.Q2gen(space);
            break;
        case JMP_LU:
            Cs.Csgen(space);
            break;
        case JMP_LAPACK:
            LA.LAgen(space);
            break;
    }
    
    // Create c^\kappa
    LSCMint kappaoff = 0;
    for (LSCMint i = 0; i < m; ++i) {
        if (!D[i]) kappaoff += N;
        else {
            // Create c^\kappa
            VectorXd ckappa((N+1)*n);
            LSCMint joff = 0;
            LSCMint jjoff = kappaoff;
            for (LSCMint j = 0; j < n; ++j) {
                ckappa.segment(joff,N+1) = coeffs.segment(jjoff,N+1);
                joff += N+1;
                jjoff += nun;
            }
            
            // Update c^\kappa
            switch (version) {
                case JMP_SPQR:
                    ckappa = Q2.Q2solve(ckappa);
                    break;
                case JMP_LU:
                    ckappa -= Cs.Csget().transpose()*Cs.Cssolve(Cs.Csget()*ckappa);
                    break;
                case JMP_LAPACK:
                    ckappa -= LA.Csget().transpose()*LA.LAsolve(LA.Csget()*ckappa);
                    break;
            }
            
            // Restore c^\kappa
            joff = 0;
            jjoff = kappaoff;
            for (LSCMint j = 0; j < n; ++j) {
                coeffs.segment(jjoff,N+1) = ckappa.segment(joff,N+1);
                joff += N+1;
                jjoff += nun;
            }
            
            kappaoff += N+1;
        }
    }
    
    LA.LAfree();  // Avoid memory leak
}

VectorXd LSCM::GridFkt::jump(LSCMint j) const {
    if ((j <= 0) || (j >= n)) {
        cerr << "Error GridFkt: Index out of bounds" << endl;
        exit(1);
    }
    LSCMint joff = j*nun;
    LSCMint joff1 = joff-nun;
    
    VectorXd res = VectorXd::Zero(m);
    VectorXd psil = space->evall(j);
    VectorXd psir = space->evalr(j);

    LSCMint loff = 0;
    for (LSCMint i = 0; i < m; i++)
        if (D[i]) {
            for (LSCMint l = 0; l <= N; l++)
                res(i) += coeffs(joff+loff+l)*psir(loff+l)-coeffs(joff1+loff+l)*psil(loff+l);
            loff += N+1;
        }
        else {
            for (LSCMint l = 0; l < N; l++)
                res(i) += coeffs(joff+loff+l)*psir(loff+l)-coeffs(joff1+loff+l)*psil(loff+l);
            loff += N;
        }
    return res;
}

GridFkt& LSCM::GridFkt::operator=(const GridFkt& g) {
    if (this != &g) {
        grid = g.grid;
        space = g.space;
        coeffs = g.coeffs;
        D = g.D;
        alldofs = g.alldofs;
        n = g.n;
        m = g.m;
        k = g.k;
        N = g.N;
        nun = g.nun;
    }
    return *this;
}

GridFkt LSCM::GridFkt::operator+(const GridFkt& g) const {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        return GridFkt(space,static_cast<VectorXd>(coeffs+g.coeffs));
    }
    else {
        GridFkt proj = GridFkt(space,g);
        return GridFkt(space,static_cast<VectorXd>(coeffs+proj.coeffs));
    }
}

GridFkt LSCM::GridFkt::operator-(const GridFkt& g) const {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        return GridFkt(space,static_cast<VectorXd>(coeffs-g.coeffs));
    }
    else {
        GridFkt proj = GridFkt(space,g);
        return GridFkt(space,static_cast<VectorXd>(coeffs-proj.coeffs));
    }
}

GridFkt LSCM::GridFkt::operator*(const GridFkt& g) const {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        return GridFkt(space,static_cast<VectorXd>(coeffs.array()*g.coeffs.array()));
    }
    else {
        GridFkt proj = GridFkt(space,g);
        return GridFkt(space,static_cast<VectorXd>(coeffs.array()*proj.coeffs.array()));
    }
}

GridFkt LSCM::GridFkt::operator/(const GridFkt& g) const {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        return GridFkt(space,static_cast<VectorXd>(coeffs.array()/g.coeffs.array()));
    }
    else {
        GridFkt proj = GridFkt(space,g);
        return GridFkt(space,static_cast<VectorXd>(coeffs.array()/proj.coeffs.array()));
    }
}

GridFkt LSCM::GridFkt::operator*(const double al) const {
    return GridFkt(space,static_cast<VectorXd>(al*coeffs));
}

GridFkt LSCM::operator*(const double al, const GridFkt& g) {
    return GridFkt(g.space,static_cast<VectorXd>(al*g.coeffs));
}

GridFkt& LSCM::GridFkt::operator+=(const GridFkt& g) {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        coeffs += g.coeffs;
    }
    else {
        GridFkt proj = GridFkt(space,g);
        coeffs += proj.coeffs;
    }
    return *this;
}
  
GridFkt& LSCM::GridFkt::operator*=(const GridFkt& g) {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        coeffs.array() *= g.coeffs.array();
    }
    else {
        GridFkt proj = GridFkt(space,g);
        coeffs.array() *= proj.coeffs.array();
    }
    return *this;
}

GridFkt& LSCM::GridFkt::operator-=(const GridFkt& g) {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        coeffs -= g.coeffs;
    }
    else {
        GridFkt proj = GridFkt(space,g);
        coeffs -= proj.coeffs;
    }
    return *this;
}

GridFkt& LSCM::GridFkt::operator/=(const GridFkt& g) {
    check_consistency(g);
    if (isSpaceIdentical(g)) {
        coeffs.array() *= g.coeffs.array();
    }
    else {
        GridFkt proj = GridFkt(space,g);
        coeffs.array() /= proj.coeffs.array();
    }
    return *this;
}

GridFkt LSCM::GridFkt::operator*(const VectorXd scal) const {
    if (scal.size() != m) {
        cerr << "GridFkt: Scaling vector has wrong dimension!" << endl;
        exit(1);
    }
    GridFkt tmp = GridFkt(space);
    LSCMint joff = 0;
    for (LSCMint j = 0; j < n; ++j) {
        for (LSCMint i = 0; i < m; ++i) {
            for (LSCMint l = 0; l < N; ++l) tmp.coeffs(joff+l) = scal(i)*coeffs(joff+l);
            joff += N;
            if (D[i]) {
                tmp.coeffs(joff) = scal(i)*coeffs(joff);
                ++joff;
            }
        }
    }
    return tmp;
}

VectorXd LSCM::GridFkt::eval(const double t) const {
    LSCMint j = grid->interv(t)-1;

    LSCMint joff = j*nun;

    VectorXd res = VectorXd::Zero(m);
    VectorXd psil = space->eval(t);

    LSCMint loff = 0;
    for (LSCMint i = 0; i < m; i++)
        if (D[i]) {
            for (LSCMint l = 0; l <= N; l++) res(i) += coeffs(joff+loff+l)*psil(loff+l);
            loff += N+1;
        }
        else {
            for (LSCMint l = 0; l < N; l++) res(i) +=coeffs(joff+loff+l)*psil(loff+l);
            loff += N;
        }
    return res;
}

VectorXd LSCM::GridFkt::deval(const double t) const {
    LSCMint j = grid->interv(t)-1;

    LSCMint joff = j*nun;

    VectorXd res = VectorXd::Zero(m);
    VectorXd psil = space->deval(t);

    LSCMint loff = 0;
    for (LSCMint i = 0; i < m; i++)
        if (D[i]) {
            for (LSCMint l = 0; l <= N; l++) res(i) += coeffs(joff+loff+l)*psil(loff+l);
            loff += N+1;
        }
        else {
            // for (LSCMint l = 0; l < N; l++) res(i) +=coeffs(joff+loff+l)*psil(loff+l);
            loff += N;
        }
    return res;
}


double LSCM::GridFkt::l2norm() const {
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            VectorXd sol = eval(tj1+sw[i].zero*hj);
            double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) sumll += sol(j)*sol(j);
            double w = sw[i].weight;
            suml += w*sumll;
        }
        sum += hj*suml;
    }
    return sqrt(sum);
}

double LSCM::GridFkt::h1norm() const {
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd sol = eval(tl);
            VectorXd dsol = deval(tl);
            double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) sumll += sol(j)*sol(j)+dsol(j)*dsol(j);
            double w = sw[i].weight;
            suml += w*sumll;
        }
        sum += hj*suml;
    }
    
    return sqrt(sum);
}

VectorXd LSCM::GridFkt::cl2norm() const {
    VectorXd sum = VectorXd::Zero(m);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        VectorXd suml = VectorXd::Zero(m);
        for (LSCMint i = 0; i < M; ++i) {
            VectorXd sol = eval(tj1+sw[i].zero*hj);
            double w = sw[i].weight;
            for (LSCMint j = 0; j < m; ++j)
                 suml(j) += w*sol(j)*sol(j);
        }
        for (LSCMint j = 0; j < m; j++) sum(j) += hj*suml(j);
    }
    return sum.array().sqrt();
}

VectorXd LSCM::GridFkt::cinfnorm() const {
    VectorXd sum = VectorXd::Zero(m);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        for (LSCMint i = 0; i < M; ++i) {
            VectorXd sol = eval(tj1+sw[i].zero*hj);
            for (LSCMint j = 0; j < m; ++j)
                 if (sum(j) < abs(sol(j))) sum(j) = abs(sol(j));
        }
    }
    return sum;
}

double LSCM::GridFkt::infnorm() const {
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        for (LSCMint i = 0; i < M; ++i) {
            VectorXd sol = eval(tj1+sw[i].zero*hj);
            for (LSCMint j = 0; j < m; ++j)
                 if (sum < abs(sol(j))) sum = abs(sol(j));
        }
    }
    return sum;
}

double LSCM::GridFkt::l2dist(function<VectorXd(double)> exsol) const {
    VectorXd res = exsol((*grid)[0]);
    if (m != res.size()) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
    }
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd diff = eval(tl)-exsol(tl);
            double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) sumll += diff(j)*diff(j);
            double w = sw[i].weight;
            suml += w*sumll;
        }
        sum += hj*suml;
    }
    return sqrt(sum);
}

double LSCM::GridFkt::h1dist(function<VectorXd(double)> exsol,
                       function<VectorXd(double)> diffexsol) const {
    VectorXd res = exsol((*grid)[0]);
    if (m != res.size()) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
    }
    res = diffexsol((*grid)[0]);
    if (m != res.size()) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
    }
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd diff1 = eval(tl)-exsol(tl);
            VectorXd diff2 = deval(tl)-diffexsol(tl);
           double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) {
                sumll += diff1(j)*diff1(j);
                if (D[j]) sumll += diff2(j)*diff2(j);
            }
            double w = sw[i].weight;
            suml += w*sumll;
        }
        sum += hj*suml;
    }
    return sqrt(sum);
}

VectorXd LSCM::GridFkt::cl2dist(function<VectorXd(double)> exsol) const {
    VectorXd res = exsol((*grid)[0]);
    if (m != res.size()) {
            cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
        }
    VectorXd sum = VectorXd::Zero(m);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        VectorXd suml = VectorXd::Zero(m);
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd diff = eval(tl)-exsol(tl);
            double w = sw[i].weight;
            for (LSCMint j = 0; j < m; ++j)
                 suml(j) += w*diff(j)*diff(j);
        }
        for (LSCMint j = 0; j < m; j++) sum(j) += hj*suml(j);
    }
    return sum.array().sqrt();
}

double LSCM::GridFkt::infdist(function<VectorXd(double)> exsol) const {
    VectorXd res = exsol((*grid)[0]);
    if (m != res.size()) {
            std::cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
        }
    double sum = 0.0;
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd diff = eval(tl)-exsol(tl);
            for (LSCMint j = 0; j < m; ++j)
                 if (sum < abs(diff(j))) sum = abs(diff(j));
        }
    }
    return sum;
}

VectorXd LSCM::GridFkt::cinfdist(function<VectorXd(double)> exsol) const {
    VectorXd res = exsol((*grid)[0]);
    if (m != res.size()) {
            std::cout << "GridFkt: Incompatible functions" << endl;
            exit(1);
        }
    VectorXd sum = VectorXd::Zero(m);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd diff = eval(tl)-exsol(tl);
            for (LSCMint j = 0; j < m; ++j)
                 if (sum(j) < abs(diff(j))) sum(j) = abs(diff(j));
        }
    }
    return sum;
}

VectorXd LSCM::GridFkt::h1locnorm() const {
    VectorXd res = VectorXd::Zero(n);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd sol = eval(tl);
            VectorXd dsol = deval(tl);
            double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) sumll += sol(j)*sol(j)+dsol(j)*dsol(j);
            double w = sw[i].weight;
            suml += w*sumll;
        }
        res(nn) =  hj*suml;
    }
    return res.array().sqrt();
}

VectorXd LSCM::GridFkt::l2locnorm() const {
    VectorXd res = VectorXd::Zero(n);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        double suml = 0.0;
        for (LSCMint i = 0; i < M; ++i) {
            double tl = tj1+sw[i].zero*hj;
            VectorXd sol = eval(tl);
            double sumll = 0.0;
            for (LSCMint j = 0; j < m; ++j) sumll += sol(j)*sol(j);
            double w = sw[i].weight;
            suml += w*sumll;
        }
        res(nn) =  hj*suml;
    }
    return res.array().sqrt();
}

VectorXd LSCM::GridFkt::inflocnorm() const {
    VectorXd res = VectorXd::Zero(n);
    const QuadratureRule::IntMethod& sw = space->getsqrtw();
    LSCMint M = sw.size();
    for (LSCMint nn = 0; nn < n; ++nn) {
        double sum = 0.0;
        double tj1 = (*grid)[nn];
        double tj = (*grid)[nn+1];
        double hj = tj-tj1;
        for (LSCMint i = 0; i < M; ++i) {
            VectorXd sol = eval(tj1+sw[i].zero*hj);
            for (LSCMint j = 0; j < m; ++j)
                if (sum < abs(sol(j))) sum = abs(sol(j));
        }
        res(nn) = sum;
    }
    return res;
}
/*
GridFkt LSCM::GridFkt::mismatch() const {
    VectorXd ncoeffs(coeffs.size());
    LSCMint joff = 0;
    bool even = false;
    for (LSCMint j = 0; j < n; ++j) {
        if (even) ncoeffs.segment(joff,nun) = VectorXd::Zero(nun);
        else ncoeffs.segment(joff,nun) = VectorXd::Ones(nun);
        joff += nun;
        even = !even;
    }
    return GridFkt(space,ncoeffs);
}
*/
