/* File: Grid.hpp
 *
 * This is a simple implementation of a grid. It seems to be too simple to
 * have a class for it. However, it is convenient ;-)
 * 
 * opyright (C) Michael Hanke 2018
 * Version: 2018-08-20
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef GRID_HPP
#define GRID_HPP

#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <vector>

namespace LSCM {

/**
 * This is a simple implementation of a grid. It seems to be too simple to
 * have a class for it. However, it is convenient ;-)
 */
class Grid {
private:
    std::vector<double> T;
    // Phi must be scaled such that
    //    \sum_{i=1}^n \Phi_i^{-1} = n
    Eigen::VectorXd Phi;
    LSCMint n;
    // This variable is for optimizing the runtime: If an interval has been
    // determined that holds a given t, the next query is often nearby. This
    // requires to save the state of previous queries.
    mutable LSCMint interv_state = 1;
    
public:
    /**
     * Constructor
     * 
     * Creates an equidistant grid on the interval [a,b]
     * 
     * @param[in] a left boundary of interval
     * @param[in] b rightboundary
     * @param[in] n number of subintervals
     */
    Grid(double a = 0.0, double b = 1.0, LSCMint n = 1);
    
    /**
     * Constructor
     * 
     * @param[in] T creates a grid using t_i
     */
    Grid(std::vector<double>& T);
    
    /**
     * Constructor
     * 
     * @param[in] Phi creates a grid using Phi_i+1/2
     */
    Grid(double a, double b, Eigen::VectorXd& Phi);
    
    /**
     * Indexing operator
     * 
     * Note that there is no index checking implemented. It is the responsibility
     * of the caller to do that!
     * 
     * @returns the i-th grid point, 0 <= i <= n
     */
    double operator[](LSCMint i) const { return T[i]; }
    
    /**
     * Access to Phi
     * 
     * Note that there is no index checking implemented. It is the responsibility
     * of the caller to do that!
     * 
     * @returns the i-th grid point, 0 <= i < n
     */
    double phi(LSCMint i) const { return Phi(i); }
    
    /**
     * Access to Phi
     * 
     * @returns vector Phi
     */
    const Eigen::VectorXd& getphi() { return Phi; }
    
    /**
     * Locate interval where t belongs to. Note that the return value is also
     * cached.
     * 
     * @param[in] t find the subinterval where t belongs to
     * @returns the index such that T[i-1] < t <= T[i]
     */
    LSCMint interv(double t);
    
    /**
     * queries for the grid size
     * 
     * @returns number of subintervals of the current grid
     */
    LSCMint getn() const { return n; }
};

}

#endif
