 /* File: Grid.hpp
 *
 * This is a simple implementation of a grid. It seems to be too simple to
 * have a class for it. However, it is convenient ;-)
 * 
 * This is the impementation
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2018-08-20
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Grid.hpp"
#include <Eigen/Dense>
#include <vector>
#include <iostream>
#include <cstdlib>

using namespace Eigen;
using namespace std;

LSCM::Grid::Grid(double a, double b, LSCMint n_) : n(n_) {
    // Check for valid parameters
    if (a >= b) {
        cerr << "Error Grid: Invalid interval" << endl;
        exit(1);
    }
    if (n < 1) {
        cerr << "Error Grid: Invalid number of subintervals" << endl;
        exit(1);
    }
    
    // Generate equidistant grid
    double h = (b-a)/n;
    T = vector<double>(n+1);
    for (LSCMint i = 0; i <= n; i++) 
        T[i] = i*h+a;
    Phi = VectorXd::Ones(n);
}

LSCM::Grid::Grid(vector<double>& T_) : T(T_) {
    // Check for valid parameters
    if (T.size() <= 1) {
        cerr << "Error Grid: Empty grid provided" << endl;
        exit(1);
    }
    n = T.size()-1;
    for (LSCMint i = 0; i < n; i++)
        if (T[i] >= T[i+1]) {
            cerr << "Error Grid: Grid not monotonic" << endl;
            exit(1);
        }
    Phi = VectorXd(n);
    double h = (T[n]-T[0])/n;
    for (LSCMint i = 0; i < n; ++i) Phi(i) = h/(T[i+1]-T[i]);
}

LSCM::Grid::Grid(double a, double b, VectorXd& Phi_) : Phi(Phi_) {
    // Check for valid parameters
    if (Phi.size() < 1) {
        cerr << "Error Grid: Empty grid provided" << endl;
        exit(1);
    }
    n = Phi.size();
    for (LSCMint i = 0; i < n; i++)
        if (Phi[i] <= 0.0) {
            cerr << "Error Grid: Grid density not positive" << endl;
            exit(1);
        }
    T = vector<double>(n+1);
    double h = (b-a)/n;
    T[0] = a;
    for (LSCMint i = 0; i < n-1; ++i) T[i+1] = h/Phi(i)+T[i];
    T[n] = b;
}

LSCM::LSCMint LSCM::Grid::interv(double t)
{
    // This algorithm is inspired by Carl de Boor: A practical guide to splines
    //Check for reasonable input
    if (t < T[0] || t > T[n]) {
        cerr << "Error Grid: Location poLSCMint out of range, t = " << t << endl;
        exit(1);
    }
    if (t == T[0]) {
        interv_state = 1;
        return 1;
    }

    // Start interval search
    LSCMint ihi = interv_state;
    LSCMint ilo = ihi-1;
    if (ilo < 0 || ihi > n) {
        ilo = 0;
        ihi = n;
    }
    else {
        if ( t <= T[ilo]) {
            // Shift interval to the left
            LSCMint istep = 1;
            while (true) {    // guaranteed to stop because of previous tests!
                ihi = ilo;
                ilo = ihi-istep;
                if (ilo < 0) {
                    ilo = 0;
                    break;
                }
                if (t <= T[ilo]) istep *= 2;
                else break;
            }
        }
        else if (t <= T[ihi]) {
            // Found!
            interv_state = ihi;
            return ihi;
        }
        else {
            // Shift interval to the right
            LSCMint istep = 1;
            while (true) {   // guaranteed to stop because of previous tests!
                ilo = ihi;
                ihi = ilo+istep;
                if (ihi > n) {
                    ihi = n;
                    break;
                }
                if (t > T[ihi]) istep *= 2;
                else break;
            }
        }
    }

    // At this poLSCMint we know T(ilo) < t <= T(ihi)
    LSCMint middle = (ilo+ihi)/2;
    while (middle > ilo) {   // uses properties of integer division: Rounding against zero!!
        if (t <= T[middle]) ihi = middle;
        else ilo = middle;
        middle = (ilo+ihi)/2;
    }

    // Done
    interv_state = ihi;
    return ihi;
}
