/* File: GnuPlot.hpp
 *
 * This class provides a simple interface to gnuplot for plotting gridfunctions.
 * The idea is to mimic some functionality from matlab plotting routines.
 * 
 * Copyright (C) Michael Hanke 2021
 * Version: 20121-05-17
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef GNUPLOT_HPP
#define GNUPLOT_HPP

#include "LSCMConfig.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include <vector>
#include <deque>
#include <iostream>
#include <string>
#include <list>
#include <cstdio>

namespace LSCM {

/**
 * This class provides basic plotting functionality. The functionality is motivated
 * by the matlab philosophy. The plotting backend is gnuplot.
 */
class GnuPlot {
private:
    std::string title_;
    std::string xlabel_;
    std::string ylabel_;
    std::string device_;
    LSCMint nplot_ = 101;
    
    typedef struct { Eigen::VectorXd tlist;
        Eigen::MatrixXd ylist;
        std::vector<std::string> ylabels;
    } Plotitem;
    std::list<Plotitem> plotlist;
    
    FILE *to_gnuplot;
    
    void initdev();
    
public:
    /**
     * Type for describing the axes scalings
     */
    typedef enum { PLAIN,   /**< linear scaling of both axes */
                   SEMILOGX,/**< logarithmic scaled x-axis */
                   SEMILOGY,/**< logarithmic scaled y-axis */
                   LOGLOG   /**< both axes are logarithmically scaled */
    } axisstyle;
    
    /**
     * Default constructor
     */
    GnuPlot() { initdev(); }
    
    /**
     * Constructor using an alternative output device
     * 
     * @param[in] device output device to use
     */
    GnuPlot(std::string device) : device_(device) {
        std::cerr << "Changing device not yet implemented. Use default device" << std::endl;
        initdev();
    }
    
    /**
     * Plot of y vs t. If t had dimension n, y must have the dimensions n x m.
     * The integer vector comp can be used to plot individual components,
     * y(.,comp(j)). It must hold 0 <= comp(j) < m. If comp is empty (default),
     * all components of y will be plotted.
     * 
     * @param[in] t vector of abscissa values
     * @param[in] y matrix of ordinates
     * @param[in] comp indication of components to be plottet
     * @param[in] labels to be used for naming curves
     */
    void plot(const Eigen::VectorXd& t, const Eigen::MatrixXd& y,
         const std::vector<int>& comp = {},
         const std::vector<std::string>& labels = {});
    
    /**
     * Plot of the grid function x. 
     * The integer vector comp can be used to plot individual components,
     * x(.,comp(j)). It must hold 0 <= comp(j) < x.cols(). If comp is empty (default),
     * all components of x will be plotted.
     * 
     * @param[in] x grid function to be plotted
     * @param[in] comp indication of components to be plottet
     * @param[in] labels to be used for naming curves
     */
    void plot(const GridFkt& x, const std::vector<int>& comp = {},
         const std::vector<std::string>& labels = {});
    
    /**
     * Plot of the grid function x. 
     * The integer vector comp can be used to plot individual components,
     * x(.,comp(j)). It must hold 0 <= comp(j) < x.cols(). If comp is empty (default),
     * all components of x will be plotted.
     * 
     * @param[in] x grid function to be plotted (output of time stepper)
     * @param[in] comp indication of components to be plottet
     * @param[in] labels to be used for naming curves
     */
    void plot(const std::shared_ptr<std::deque<GridFkt>> x, 
              const std::vector<int>& comp = {},
              const std::vector<std::string>& labels = {});
    
    void setTitle(const std::string& title) { title_ = title; }
    
    void setXlabel(const std::string& xlabel) { xlabel_ = xlabel; }
    
    void setYlabel(const std::string& ylabel) { ylabel_ = ylabel; }
    
    void setNplot(const LSCMint nplot) { nplot_ = nplot; }
    
    /**
     * Generate the output. Any output generated by previous calls to show() will
     * be deleted.
     */
    void show(axisstyle style = PLAIN);
    
    /**
     * Clear the curve data
     */
    void clear() { plotlist.clear(); }
    
    /**
     * Destructor: Note that the output pipeline will be closed such that also the
     * plots will be closed. A non-persistent plot will be erased (depends on device).
     */
    ~GnuPlot();
};

}

#endif
