/* File: GnuPlot.hpp
 *
 * This class provides a simple interface to gnuplot for plotting gridfunctions.
 * The idea is to mimic some functionality from matlab plotting routines.
 * 
 * Copyright (C) Michael Hanke 2021
 * Version: 2021-05-17
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "GnuPlot.hpp"
#include <Eigen/Dense>
#include <vector>
#include <deque>
#include <iostream>
#include <string>
#include <cstdio>
#include <list>
#include <cstdlib>

using namespace std;
using namespace Eigen;

void LSCM::GnuPlot::initdev() {
    to_gnuplot = popen("gnuplot -p >/dev/null","w");
    // to_gnuplot = stderr;
    if (!to_gnuplot) {
        cerr << "Could not start gnuplot!" << endl;
        exit(1);
    }
}

void LSCM::GnuPlot::plot(const VectorXd& t, const MatrixXd& y, const vector<int>& comp,
                   const vector<string>& labels) {
    // Check consistency
    if ((t.size() != y.rows()) || (t.size() == 0)) {
        cerr << "Plot data size mismatch. Ignored." << endl;
        return;
    }
    
    Plotitem item;
    item.tlist = t;
    item.ylabels = labels;
    if (comp.size() == 0) item.ylist = y;
    else {
        item.ylist = MatrixXd::Zero(t.size(),comp.size());
        for (LSCMint i = 0; i < comp.size(); ++i) {
            if ((comp[i] < 0) || (comp[i] >= y.cols())) {
                cerr << "Component selection invalid. Ignored" << endl;
                return;
            }
            item.ylist.col(i) = y.col(comp[i]);
        }
    }
    
    plotlist.push_back(item);
}

void LSCM::GnuPlot::plot(const GridFkt& x, const vector<int>& comp,
                   const vector<string>& labels) {
    double a = x.geta();
    double b = x.getb();
    LSCMint m = x.getm();

    Plotitem item;
    item.tlist = VectorXd::LinSpaced(nplot_,a,b);
    item.ylabels = labels;
    LSCMint ycol = comp.size() == 0 ? m : comp.size();
    item.ylist = MatrixXd::Zero(nplot_,ycol);
    for (LSCMint j = 0; j < nplot_; ++j) {
        VectorXd tmp = x(item.tlist(j));
        if (comp.size() == 0) item.ylist.row(j) = tmp;
        else {
            for (LSCMint i = 0; i < comp.size(); ++i) {
                if ((comp[i] < 0) || (comp[i] >= m)) {
                    cerr << "Component selection invalid. Ignored" << endl;
                    return;
                }
                item.ylist(j,i) = tmp(comp[i]);
            }
        }
    }
    
    plotlist.push_back(item);
}

void LSCM::GnuPlot::plot(const shared_ptr<deque<GridFkt>> x, 
              const vector<int>& comp, const vector<string>& labels) {
    double a = (x->begin())->geta();
    LSCMint lb = x->size()+1;
    VectorXd breakpoints(lb);
    breakpoints(0) = a;
    LSCMint i = 0;
    for (auto &xi : *x) breakpoints(++i) = xi.getb();
    double b = breakpoints(lb-1);
    for (LSCMint i = 1; i < lb; ++i)
        if (breakpoints(i-1) > breakpoints(i)) {
            // Should never happen!
            cerr << "GnuPlot: breakpoints not monotonically increasing" << endl;
            exit(1);
        }
    LSCMint m = (x->begin())->getm();
    
    Plotitem item;
    item.tlist = VectorXd::LinSpaced(nplot_,a,b);
    item.ylabels = labels;
    LSCMint ycol = comp.size() == 0 ? m : comp.size();
    item.ylist = MatrixXd::Zero(nplot_,ycol);
    
    LSCMint bind = 1;
    auto xelem = x->begin();
    for (LSCMint j = 0; j < nplot_; ++j) {
        double tj = item.tlist(j);
        while (tj > breakpoints(bind)) {
            ++bind;
            ++xelem;
        }
        VectorXd tmp = (*xelem)(item.tlist(j));
        if (comp.size() == 0) item.ylist.row(j) = tmp;
        else {
            for (LSCMint i = 0; i < comp.size(); ++i) {
                if ((comp[i] < 0) || (comp[i] >= m)) {
                    cerr << "Component selection invalid. Ignored" << endl;
                    return;
                }
                item.ylist(j,i) = tmp(comp[i]);
            }
        }
    }
    
    plotlist.push_back(item);
}

void LSCM::GnuPlot::show(axisstyle style) {
    // Error handling should be added!
    // Initialize plot
    fprintf(to_gnuplot,"reset\n");
    fprintf(to_gnuplot,"set style data lines\n");
    switch (style) {
        case PLAIN:
            break;
        case SEMILOGX:
            fprintf(to_gnuplot,"set logscale x\n");
            break;
        case SEMILOGY:
            fprintf(to_gnuplot,"set logscale y\n");
            break;
        case LOGLOG:
            fprintf(to_gnuplot,"set logscale xy\n");
    }
    fprintf(to_gnuplot,"set colorsequence classic\n");
    if (!title_.empty())
        fprintf(to_gnuplot,"set title \"%s\"\n",title_.c_str());
    if(!xlabel_.empty())
        fprintf(to_gnuplot,"set xlabel \"%s\"\n",xlabel_.c_str());
    if(!ylabel_.empty())
        fprintf(to_gnuplot,"set ylabel \"%s\"\n",ylabel_.c_str());
    
    // Generate curve data
    // t axis should be identical! Shall we check it??
    if (plotlist.size() == 0) {
        cerr << "Nothing to plot!" << endl;
        return;
    }
    // Use datablocks
    LSCMint ncurves = 0;
    for (auto &item : plotlist) {
        for (LSCMint i = 0; i < item.ylist.cols(); ++i) {
            fprintf(to_gnuplot,"$line_%d << EOD\n",ncurves);
            ++ncurves;
            for (LSCMint j = 0; j < item.tlist.size(); ++j)
                fprintf(to_gnuplot,"%15.10e %15.10e\n",item.tlist(j),item.ylist(j,i));
            fprintf(to_gnuplot,"EOD\n");
        }
    }
    LSCMint nc = 0;
    fprintf(to_gnuplot,"plot ");
    for (auto &item : plotlist) {
        for (LSCMint i = 0; i < item.ylist.cols(); ++i) {
            if (nc > 0) fprintf(to_gnuplot,", ");
            fprintf(to_gnuplot,"$line_%d",nc);
            if ((item.ylabels.size() > i) && (!item.ylabels[i].empty()))
                fprintf(to_gnuplot," title \"%s\"",item.ylabels[i].c_str());
            ++nc;
        }
    }
    fprintf(to_gnuplot,"\n");
    if (ncurves != nc) {
        cerr << "Something is strange!" << endl;
    }
    
    fprintf(to_gnuplot,"\n");
}

LSCM::GnuPlot::~GnuPlot() {
    if (pclose(to_gnuplot) != 0) {
        cerr << "Error stopping gnuplot!" << endl;
        exit(1);
    }
}
