/* File: GaussLegendre.cpp
 *
 * Class implementing the Lobatto-Legendre integration
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "GaussLegendre.hpp"
#include "gsl_integration_mod.h"
#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <vector>
#include <functional>
#include <algorithm>
#include <iostream>
#include <cstdlib>
#include <limits>

using namespace LSCM;
using namespace Eigen;
using namespace std;


void GaussLegendre::generate() {
    //   This method is based on a revision of glfixed.c in:
    //   Gnu Scientific Library, https://www.gnu.org/software/gsl
    gsl_integration_glfixed_table* tbl =
            gsl_integration_glfixed_table_alloc (n_);
        
    VectorXd xt(n_);
    VectorXd wt(n_);
    for (LSCMint i = 0; i < n_; ++i)
        gsl_integration_glfixed_point(-1.0,1.0,i,xt.data()+i,wt.data()+i,tbl);
    gsl_integration_glfixed_table_free(tbl);

    // Sorting necessary? It is not quite clear if the nodes are provided in increasing
    // order. Since quicksort works worst if the keys are already sorted, we make n extra check.
    vector<LSCMint> index(n_);
    bool sorted = true;
    index[0] = 0;
    for (LSCMint i = 1; i < n_; ++i) {
        index[i] = i;
        if (xt(i) <= xt(i-1)) sorted = false;
    }
    if (!sorted) {
        // The following version must be used in since both xt and wt must be 
        // sorted while the contents of xt is the keys
        // NOTE: Would it be better to create a PermutationMatrix<LSCMint>?
        auto cpfkt = [&xt](const LSCMint& i, const LSCMint& j) -> bool {
            return (xt(i) < xt(j));
        };
        sort(index.begin(),index.end(),cpfkt);
    }
    x = VectorXd(n_);
    w = VectorXd(n_);
    for (LSCMint i = 0; i < n_; ++i) {
        x(i) = xt(index[i]);
        w(i) = wt(index[i]);
    }

    c = sqrt(((VectorXd::Constant(n_,1.0)-
                    x.cwiseProduct(x)).cwiseProduct(w)).array());
    double cmax = c.maxCoeff();
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    c *= (1.0/cmax);
    baryScale = bscaljac(n_-1,0.0,0.0)*cmax;
    
    t = VectorXd::Zero(0);

    RuleProperties = QR_SYMMETRIC;
    if (n_%2 == 1) RuleProperties = RuleProperties | QR_CENTERED;
}
