/* File: DeferredCorrectionSolver.hpp
 *
 * Realization of Barlow's version of iterative improvement for linear DAEs
 * for a
 * given grid and approximation space.
 * 
 * Copyright (C) Michael Hanke 2020
 * Version: 2022-06-05
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef DEFERREDCORRECTIONSOLVER_HPP
#define DEFERREDCORRECTIONSOLVER_HPP

#include "LSCMConfig.hpp"
#include "LinLSQSolver.hpp"
#include "GridFkt.hpp"
#include <Eigen/Dense>
#include "Eigen/SPQRSupport"
#include <iostream>
#include <cstdlib>

namespace LSCM {

/**
 * Implementation of the classic version of the deferred correction solver
 * 
 * This class provides the deferred correction solver as proposed by Barlow
 * 
 */
class DeferredCorrectionSolver : public LinLSQSolver {
private:
    double tol = 1e-15;
    double kmax = 5;  // at most 2 iterations should be sufficient!
    // weight for continuity conditions
    // There are different recommendations found in the literature:
    // Charles van Loan (1985): (macheps)^(-1/2)
    // Jesse L. Barlow (1992): (macheps)^(-1/3)
    double weight = std::pow(std::numeric_limits<double>::epsilon(),-1.0/3.0);

    LSCMSparseMatrix X;
    // OBS: Using pointers to patch a bug in
    // ~Eigen::SPQR<Eigen::SparseMatrix<double>>()
    Eigen::SPQR<LSCMSparseMatrix>* qrX = nullptr;
    LSCMint Xrows, Arows, Crows, Acols;
    // Memory management for genmatT()
    LSCMindex *Xcolptr = nullptr;
    LSCMindex *Xrowidx = nullptr;
    double *Xvals = nullptr;
    
    void resetLocalVars() {
        delete qrX;
        delete [] Xrowidx;
        delete [] Xcolptr;
        delete [] Xvals;
        qrX = nullptr;
        Xrowidx = nullptr;
        Xcolptr = nullptr;
        Xvals = nullptr;
    }
    
    virtual void factorizeExec();
    
    virtual GridFkt solveExec(const Eigen::VectorXd& f);

public:
    /**
     * Default constructor
     */
    DeferredCorrectionSolver() {}
    
    /**
     * Constructor for the deferred correction solver class
     * 
     * The weight is set to the value of \f$\epsilon_{\textrm{mach}}^{-1/3}\f$.
     * 
     * \param[in] genmat matrix generation object
     */
    DeferredCorrectionSolver(std::shared_ptr<LinLSQMatrices> genmat) :
        LinLSQSolver(genmat) {}
    
    /**
     * Constructor for the deferred correction solver class
     * 
     * \param[in] genmat matrix generation object
     * \param[in] weight wcont for the continuity conditions
     */
    DeferredCorrectionSolver(std::shared_ptr<LinLSQMatrices> genmat, double wcont) :
        LinLSQSolver(genmat) {
        if (wcont <= 0.0) {
            std::cerr << "Error DeferredCorrectionSolver: Positive weight required" << std::endl;
            exit(1);
        }
        else weight = wcont;
    }
    
    /**
     * Set the weight for the continuity conditions
     * 
     * @param[in] wcont weight;
     */
    void setWeight(const double wcont) {
        if (wcont <= 0.0) {
            std::cerr << "Error setWeight: Positive weight required. Weight not changed" << std::endl;
        }
        else weight = wcont;
    }
    
    /**
     * Get the weight for the continuity conditions
     * 
     * @returns weight for continuity conditions
     */
    double getWeight() const { return weight; }

    /**
     * Set tolerance for the iterations
     * 
     * \param[in] tolerance tolerance to be used, must be > 0. The standard value is 
     *                      1e-15.
     */
    void setTol(const double tolerance) {
        if (tolerance <= 0.0)
            std::cerr << "DeferredCorrectionSolver: Tolerance <= 0, not changed!" << std::endl;
        tol = tolerance;
    }
    
    /**
     * Set maximal number of iterations
     * 
     * \param[in] itmax Maximal number of iterations, must be > 0. The standard value
     *                  is 2.
     */
    void setItmax(const LSCMint itmax) {
        if (itmax < 0)
            std::cerr << "DeferredCorrectionSolver: itmax < 0, not changed!" << std::endl;
        kmax = itmax;
    }
    
    ~DeferredCorrectionSolver() {
        resetLocalVars();
    }
    
private:
    /**
     * Setup of system matrix
     */
    LSCMSparseMatrix genmatT();
};

}

#endif
