/* File: DAE.hpp
 *
 * This class collects the functions describing a DAE.
 *
 * The definition of the grid and the approximating space depends already
 * on the DAE.
 * 
 * Copyright (C) Michael Hanke 2018
 * Version: 2019-11-08
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef DAE_HPP
#define DAE_HPP

#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <functional>
#include <vector>

namespace LSCM {

/**
 * This class describes the interface to a DAE.
 * 
 * We use the model
 * \f[
 *   A(Dx)'(t)+B(t)x(t) = q(t),\quad G_ax(a)+G_b(x(b))=r
 * \f]
 * for \f$t\in(a,b)\f$.
 *
 * Note that The definition of the grid and the approximating space depends already
 * on the DAE!
 */
class DAE {
protected:
    LSCMint l = 0;
    LSCMint index = -1;
public:
    /**
     * Type describing the interface to matrix-valued functions
     */
    typedef std::function<Eigen::MatrixXd(double)> Matfun;
    
    /**
     * Type describing the interface to vector-valued functions
     */
    typedef std::function<Eigen::VectorXd(double)> Vecfun;
    
    /**
     * The A coefficient
     * 
     * \param[in] t poLSCMint in (a,b)
     * \returns A(t)
     */
    virtual Eigen::MatrixXd A(double t) = 0;
    
    /**
     * The B coefficient
     * 
     * \param[in] t poLSCMint in (a,b)
     * \returns B(t)
     */    
    virtual Eigen::MatrixXd B(double t) = 0;
    
    /**
     * The right-hand side q
     * 
     * \param[in] t poLSCMint in (a,b)
     * \returns q(t)
     */
    virtual Eigen::VectorXd q(double t) = 0;
    
    /**
     * The D coefficient
     * 
     * \returns diag(D)
     */
    virtual std::vector<bool> getD() = 0;

    /**
     * The \f$G_a\f$ coefficient
     * 
     * \returns \f$G_a\f$
     */
    virtual Eigen::MatrixXd getGa() = 0;

    /**
     * The \f$G_b\f$ coefficient
     * 
     * \returns \f$G_b\f$
     */
    virtual Eigen::MatrixXd getGb() = 0;
    
    /**
     * The right-hand side r
     * 
     * @returns r
     */
    virtual Eigen::VectorXd getr() = 0;
    
    /**
     * Returns number of boundary conditios
     * 
     * @returns l
     */
    virtual LSCMint getl() { return l; }
    
    /**
     * returns the index of the problem (if known)
     * 
     * @returns index
     */
    LSCMint getindex() { return index; }
    
    virtual ~DAE() = default;
};

}

#endif
