/* File: Chebyshev1.cpp
 *
 * Class implementing the Radau-Legendre integration
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Chebyshev2.hpp"
#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <unsupported/Eigen/FFT>
#include <iostream>
#include <cmath>
#include <complex>
#include <cstdlib>

using namespace LSCM;
using namespace Eigen;
using namespace std;

void Chebyshev2::generate() {
    
// special case
    if (n_ == 1) {
        x = VectorXd::Zero(1); 
        w = VectorXd::Constant(1,2.0); 
        c = VectorXd::Constant(1,1.0);
        t = VectorXd::Constant(1,0.5*M_PI);
        return;
    }

    // General case
    LSCMint m = n_ - 1;
    FFT<double> fft;
    VectorXd mom((n_+1)/2);
    for (LSCMint i = 0; i < mom.size(); ++i) mom(i) = 2.0/(1.0-4*i*i);
    x = VectorXd(n_);
    for (LSCMint i = 0; i < n_; ++i) x(i) = sin((M_PI/(2*m))*(2*i-m));

    // Quadrature weights: Follows Waldvogel with modifications from chebfun         
    VectorXcd d(m);
    d << static_cast<VectorXcd>(mom), static_cast<VectorXcd>(mom(seq(n_/2-1,1,fix<-1>)));
    VectorXcd wc(m);
    fft.inv(wc,d);
    w = VectorXd(n_);
    w.head(m) = wc.real();
    w(0) *= 0.5;
    w(m) = w(0);

    // Barycentric weights:            
    c = VectorXd::Constant(n_,1.0);
    c(n_-1) = 0.5;      // Note v(end) is positive.
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    c(0) = c(0)*0.5;
    baryScale = bscaljac(n_-3,0.5,0.5)*sqrt(M_PI/(n_-1));

    // Angles:
    t = VectorXd(n_);
    double fac = M_PI/(n_-1);
    for (LSCMint i = 0; i < n_; ++i) t(m-i) = i*fac;

    RuleProperties = QR_LEFT | QR_RIGHT | QR_SYMMETRIC;
    if (n_%2 == 1) RuleProperties = RuleProperties | QR_CENTERED;
}
