/* File: Chebyshev1.cpp
 *
 * Class implementing the Radau-Legendre integration
 * 
 * C Michael Hanke 2023
 * Version: 2023-02-13
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Chebyshev1.hpp"
#include "LSCMConfig.hpp"
#include <Eigen/Dense>
#include <unsupported/Eigen/FFT>
#include <iostream>
#include <cmath>
#include <complex>
#include <cstdlib>

using namespace LSCM;
using namespace Eigen;
using namespace std;

void Chebyshev1::generate() {
    
// special case
    if (n_ == 1) {
        x = VectorXd::Zero(1); 
        w = VectorXd::Constant(1,2.0);
        c = VectorXd::Constant(1,1.0);
        t = VectorXd::Constant(1,0.5*M_PI);
        return;
    }

    // General case
    FFT<double> fft;
    VectorXd mom((n_+1)/2);
    for (LSCMint i = 0; i < mom.size(); ++i) mom(i) = 2.0/(1.0-4*i*i);
    x = VectorXd(n_);
    for (LSCMint i = 0; i < n_; ++i) x(i) = sin((M_PI/(2*n_))*(2*i+1-n_));
            
    // Quadrature weights: Procedure copied from chebtech1
    VectorXcd d(n_);
    if (n_%2 == 1)
        d << mom, -mom(seq(last,1,fix<-1>));
    else
        d << mom, 0.0, -mom(seq(last,1,fix<-1>));
    complex<double> iunit(0.0,1.0);
    for (LSCMint i = 0; i < n_; ++i) d(i) *= exp(((M_PI/n_)*i)*iunit);
    VectorXcd wc(n_);
    fft.inv(wc,d);
    w = wc.real();
            
    // Barycentric weights:
    c = VectorXd(n_);
    for (LSCMint i = n_-1; i >= n_/2; --i) c(i) = sin((M_PI/n_)*(n_-i-0.5));
    // Improving the accuracy by flipping
    for (LSCMint i = 0; i < n_/2; ++i) c(i) = c(n_-i-1);
    for (LSCMint i = n_-2; i >= 0; i -= 2) c(i) = -c(i);
    
    // Barycentric scale factor: Not very elegant
    LSCMint j = n_/2;
    baryScale = bscaljac(n_-1,0.0,0.0)*sqrt((1.0-x(j)*x(j))*w(j))/abs(c(j));
            
    // Angles
    t = VectorXd(n_);
    for (LSCMint i = 0; i < n_; ++i) t(i) = (M_PI/n_)*(n_-i-0.5);
    
    RuleProperties = QR_SYMMETRIC;
    if (n_%2 == 1) RuleProperties = RuleProperties | QR_CENTERED;
}
