/* File: Chebyshev.cpp
 *
 * This class implements the interface of class Basis using Chebyshev
 * polynomials
 * for representing polynomials.
 * 
 * Copyright (C) Michael Hanke 2019
 * Version: 2019-12-16
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "Chebyshev.hpp"
#include <Eigen/Dense>
#include <iostream>
#include <cstdlib>
#include <vector>

using namespace std;
using namespace Eigen;

 VectorXd LSCM::Chebyshev::chebyshevP(double tau, LSCMint n) const {
    VectorXd res(n);
    res(0) = 1.0;
    if (n == 1) return res;
    double x = 2.0*tau-1.0;
    res(1) = x;
    for (LSCMint i = 2; i < n; ++i)
        res(i) = 2.0*x*res(i-1)-res(i-2);
    return res;
}

VectorXd LSCM::Chebyshev::evalPd(double tau) const {
    VectorXd res(dim+1);
    VectorXd p = chebyshevP(tau,dim+1);
    res(0) = 1.0;

    res(1) = tau;
    if (dim == 1) return res;
    res(2) = tau*(tau-1.0);
    if (dim == 2) return res;
    double trm = -2.0;
    // This approach may be questionable!
    for (LSCMint i = 3; i <= dim; ++i) {
        res(i) = (p(i)*(i-2)-p(i-2)*i+trm)/(4*i*(i-2));
        trm = -trm;
    }
    return res;
}

VectorXd LSCM::Chebyshev::evaldPd(double tau) const {
    VectorXd res(dim+1);
    res(0) = 0.0;
    res.segment(1,dim) = chebyshevP(tau,dim);
    return res;
}
