/* File: Basis.hpp
 *
 * This class defines the basic interface for basic functions. The
 * basis functions are defined on the reference interval [0,1].
 * 
 * Assumption: The number of degrees of freedom for the differential
 * components is one higher than that of the algebraic components. This
 * additional degree of freedom is used for fulfilling the continuity
 * conditions for the differential components.
 * 
 * The dimension of the space is the number of degrees of freedom of
 * the algebraic components.
 *
 * Copyright (C) Michael Hanke 2018
 * Version: 2018-08-19
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/


#ifndef BASIS_HPP
#define BASIS_HPP

#include "LSCMConfig.hpp"
#include "GaussLegendre.hpp"
#include <Eigen/Dense>
#include <memory>
#include <iostream>
#include <cstdlib>

namespace LSCM {

/**
 * Abstract class to define the interface to a polynomial basis
 * 
 * This interface constructs one element of the interpolation space for
 * the least-squares method in reference coordinates, that is on [0,1].
 * 
 * We assume that the differential components have N+1 degrees of freedom while the
 * algebraic components have N degrees of freedom. It is up to the derived classes to
 * provide a reasonable implementation of this functions.
 */
class Basis {
private:
    std::shared_ptr<GaussLegendre> qrule = nullptr;
protected:
    QuadratureRule::IntMethod sqrtweight; /** weigths of Integration formula */
    const LSCMint dim = 0;              /** algebraic degrees of freedom per base kind */
    
    /**
     * Constructor for a basis object containing evaluation routines for
     * basis functions approximating algebraic and differential cmponents.
     * 
     * @param[in] N numer of degrees of freedom for the algebraic basis
     *              functions
     */
    Basis(LSCMint N) : dim(N) {
        if (dim <= 0) {
            std::cerr << "Error Basis: N <= 0" << std::endl;
            std::exit(1);
        }

        qrule = std::make_shared<GaussLegendre>(dim+2);
        sqrtweight = qrule->getMethod();
    }
    
public:
    /**
     * Compute all algebraic basis functions at tau
     * 
     * @param[in] tau argument for evaluation, 0 <= tau <= 1
     * @returns   polynomial value
     */
    virtual Eigen::VectorXd evalPa(double tau) const = 0;
    
    /**
     * Compute all differential basis functions at tau
     * 
     * @param[in] tau argument for evaluation, 0 <= tau <= 1
     * @returns   polynomial value
     */
    virtual Eigen::VectorXd evalPd(double tau) const = 0;
    
    /**
     * Compute the derivative of all differential basis functions at tau
     * 
     * @param[in] tau argument for evaluation, 0 <= tau <= 1
     * @returns   derivative value
     */
    virtual Eigen::VectorXd evaldPd(double tau) const = 0;
    
    /**
     * Number of degrees of freedom of all algebraic components
     * 
     * This function queries the number of degrees of freedom for the algebraic
     * polynomial, that is N
     * 
     * @returns N
     */
    LSCMint getdim() const {return dim; }
    
    /**
     * Integration nodes and coefficients for the L2-norm
     * 
     * @returns nodes and square roots of Gaussian quadrature
     * 
     */
    const QuadratureRule::IntMethod& getsqrtw() { return sqrtweight; }
    
    // Destructor
    virtual ~Basis() = default;
};

}

#endif
