/* File: AccurateIC.hpp
 *
 * Implements a routine for the computation of accurate initial values
 * for linear DAEs
 * 
 * C Michael Hanke 2023
 * Version: 2023-05-14
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef ACCURATEIC_HPP
#define ACCURATEIC_HPP

#include "LSCMConfig.hpp"
#include "DAE.hpp"
#include "Chebyshev2.hpp"
#include "QuadratureRule.hpp"
#include <memory>
#include "Eigen/Dense"
#include <cstdlib>

namespace LSCM {
    
/**
 * Computation of accurate initial conditions for linear DAEs
 * 
 * This class can be used to compute the index as well as a matrix G(t)
 * having the same nullspace as the canonical subspace \f$N_{can}(t)\f$
 * using the new approach via a regular dae. As a byproduct, it can be
 * used to check given initial conditions for apprpriateness.
 */
class AccurateIC {
public:

/** This type is used to indicate at which point the output should be
 * provided. CI_LEFT and CI_RIGHT should be used for initial conditions
 * while CI_CENTER is used in the middle of the interval.
 */
    typedef enum { CI_LEFT,   /**< location at left boundary */
                   CI_RIGHT,  /**< location at right boundary */
                   CI_CENTER  /**< inside the interval */
                 } CILocation;

/** This type is used to select the orthogonalization procedure
 */
    typedef enum { CI_SVD,        /**< Use singular value decomposition */
                   CI_QR,         /**< Use an adapted QR decomposition */
                   CI_UTV,        /**< Use UTV decomposition */
#ifdef USE_RRQR
                   CI_RRQR        /**< Use rank-revealing QR decomposition */
#endif
                 } CIRank;

private:
    // Computational results
    LSCMint mu = -1;
    Eigen::MatrixXd Ga;
    
    // Attributes for the computational results
    std::shared_ptr<DAE> dae_;
    double t_;
    double H_;
    CILocation loc_;

    // Algorithmic variables
    CIRank rankdet = CI_QR;
    double sf = 1e2;
    
    // Local variables to shorten calling sequences
    LSCMint N, M;
    double a, b;
    LSCMint iloc;
    
    // Data for collocation space
    std::shared_ptr<QuadratureRule> nodes;
    Eigen::MatrixXd V, dV;
    Eigen::MatrixXd Dmat;
    
    void execComp();
    
    void initObj(LSCMint N, std::shared_ptr<QuadratureRule> nodes);
    
    void condExecComp(std::shared_ptr<DAE> dae, double t,
                      double H, CILocation loc) {
        if (mu < 0) {
            dae_ = dae;
            t_ = t;
            H_ = H;
            loc_ = loc;
            execComp();
        }
        else {
            if ((t != t_) || (H != H_) || (loc != loc_) || (dae != dae_)) {
                dae_ = dae;
                t_ = t;
                H_ = H;
                loc_ = loc;
                execComp();                
            }
        }
    }
    
    double compOpening(const Eigen::MatrixXd& Ga);
    
    /*
     * The following functions need access to private variables. Therefore,
     * they must be declared here in order to avoid unnecessary parameters.
     */
    Eigen::MatrixXd Clenshaw(const Eigen::MatrixXd& coeff);
    
    Eigen::MatrixXd dClenshaw(const Eigen::MatrixXd& coeff);
    
    Eigen::MatrixXd dC0eval(const Eigen::MatrixXd& C0);
    
    void Smoothing(Eigen::MatrixXd& U, const LSCMint m, const LSCMint r);
    
    Eigen::MatrixXd Cbasis(const std::vector<bool>& D,
                           Eigen::MatrixXd& E, Eigen::MatrixXd& F,
                           const Eigen::VectorXd& scale);
    
    void genYZ(const Eigen::VectorXd& scale, const Eigen::MatrixXd& E,
               const Eigen::MatrixXd& F,
               const LSCMint m,
               Eigen::MatrixXd& Y, Eigen::MatrixXd& Z, LSCMint& r,
               bool init);
    
public:
    /**
     * Constructor
     * 
     * The number of nodes of the quadrature rule must be larger than the
     * maximal polynomial degree.
     * 
     * \param[in] N     maximum degree of approximating polynomials
     * \param[in] nodes pointer to a quadrature rule defining the nodes
     */
    AccurateIC(LSCMint N, std::shared_ptr<QuadratureRule> nodes) {
        if (N < 1) {
            std::cerr << "AccurateIC: Polynomial degree must be positive!" << std::endl;
            std::exit(1);
        }
        initObj(N, nodes);
    }
    
    /**
     * Constructor
     * 
     * Here, M must be larger than N.
     * 
     * \param[in] N  maximum degree of approximating polynomials
     * \param[in] M  number of collocation nodes. The collocation nodes are
     *               the Chebyshev nodes of the second kind
     */
    AccurateIC(LSCMint N, LSCMint M) {
        if (N < 1) {
            std::cerr << "AccurateIC: Polynomial degree must be positive!" << std::endl;
            std::exit(1);
        }
        if (M <= N) {
            std::cerr << "AccurateIC: Number of collocation points too small!" << std::endl;
            std::exit(1);
        }
        auto nodes = std::make_shared<Chebyshev2>(M);
        initObj(N, nodes);
    }
    
    /**
     * Computes and returns a matrix \f$G_a\f$ which makes up an accurate initial
     * condition at t
     * 
     * \param[in] dae  pointer to the dae
     * \param[in] t    evaluation point
     * \param[in] H    size of the surrounding interval to be used for
     *                 numerical differentiation
     * \param[in] loc  location of t inside the surrounding interval
     * \returns   the matrix \f$G_a\f$
     */
    Eigen::MatrixXd getGa(std::shared_ptr<DAE> dae, double t,
                          double H, CILocation loc) {
        condExecComp(dae,t,H,loc);
        return Ga;
    }
    
    /**
     * Computes and returns the index of the dae at t
     * 
     * \param[in] dae  pointer to the dae
     * \param[in] t    evaluation point
     * \param[in] H    size of the surrounding interval to be used for
     *                 numerical differentiation
     * \param[in] loc  location of t inside the surrounding interval
     * \returns   the index \f$\mu\f$ of the dae
     */
    LSCMint getIndex(std::shared_ptr<DAE> dae, double t,
                     double H, CILocation loc) {
        condExecComp(dae,t,H,loc);
        return mu;
    }
    
    /**
     * Computes the opening between the nullspaces of Ga and an accurate
     * initial condition.
     * 
     * The opening is a measure for the error to be expected
     * in the the initial values at t whenusing Ga and the accurate
     * initial condition approximated in this object.
     * 
     * \param[in] Ga   the matrix to be tested
     * \param[in] dae  pointer to the dae
     * \param[in] t    evaluation point
     * \param[in] H    size of the surrounding interval to be used for
     *                 numerical differentiation
     * \param[in] loc  location of t inside the surrounding interval
     * \returns  the opening
     */
    double opening(const Eigen::MatrixXd& Ga, std::shared_ptr<DAE> dae,
                   double t, double H, CILocation loc) {
        condExecComp(dae,t,H,loc);
        return compOpening(Ga);
    }

    /**
     * Sets the procedure for rank determination
     * 
     * @param[in] rd rank determination procedure
     */
    void setRankProcedure(const CIRank rd) {
        if (rankdet == rd) return;
        mu = -1;
        rankdet = rd;
    }

    /**
     * Sets the security factor for rank determinations
     * 
     * @param[in] sf_ security factor
     */
    void setSF(const double sf_) {
        if (sf_ <= 0.0) {
            std::cerr << "AccurateIC: Security factor not changed!" << std::endl;
        }
        else {
            mu = -1;
            sf = sf_;
        }
    }
};
}

#endif
