/* Definition of the standard pendulum problem (index-3 problem) using the
 * nonlinear interface
 * 
 * Copyright (C) Michael Hanke 2019
 * Version 2019-12-17
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#ifndef NLPENDEL_HPP
#define NLPENDEL_HPP

#include "NlDAE.hpp"
#include "Eigen/Dense"
#define _USE_MATH_DEFINES
#include <cmath>

// Parameters are taken from: SL Campbell and JT Betts

static const double g = 16.0;
static const double L = std::sqrt(8.0);
static const double phi0 = M_PI_4;
static const double omega0 = 0.0;
static const double A = 0.0;
static const double B = 3.0;

class nlPendel : public LSCM::NlDAE {
private:
    const double alpha = 1.0;
public:
    nlPendel() { l = 2; }
    
    Eigen::VectorXd f(const Eigen::VectorXd& y,
                      const Eigen::VectorXd& x,
                      const double t) {
        Eigen::VectorXd res(5);
        res(0) = y(0)-x(1);
        res(1) = y(1)+x(4)*x(0);
        res(2) = y(2)-x(3);
        res(3) = y(3)+x(4)*x(2)+g;
        res(4) = x(0)*x(0)+x(2)*x(2)-L*L;
        return res;
    }
    
    Eigen::MatrixXd dfy(const Eigen::VectorXd& y,
                        const Eigen::VectorXd& x,
                        const double t) {
        Eigen::MatrixXd res(5,5);
        res << 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
                1, 0, 0, 0, 0, 0, 0;
        return res;
    }
    
    Eigen::MatrixXd dfx(const Eigen::VectorXd& y,
                        const Eigen::VectorXd& x,
                        const double t) {
        Eigen::MatrixXd res(5,5);
        res << 0, -1, 0, 0, 0,
                x(4), 0, 0, 0, x(0),
                0, 0, 0, -1, 0,
                0, 0, x(4), 0, x(2),
                2*x(0), 0, 2*x(2), 0, 0;
        return res;
    }
    
    Eigen::VectorXd r(const Eigen::VectorXd& xa,const Eigen::VectorXd& xb) {
        Eigen::VectorXd res(2);
        res(0) = xa(2)+L*cos(phi0);
        res(1) = alpha*(xa(3)-L*omega0*sin(phi0));
        return res;
    }
    
    Eigen::MatrixXd dra(const Eigen::VectorXd& xa,const Eigen::VectorXd& xb) {
        Eigen::MatrixXd Ga = Eigen::MatrixXd::Zero(2,5);
        Ga(0,2) = 1.0;
        Ga(1,3) = alpha*1.0;
        return Ga;
    }
    
    Eigen::MatrixXd drb(const Eigen::VectorXd& xa,const Eigen::VectorXd& xb) {
        return Eigen::MatrixXd::Zero(2,5);
    }
    
    std::vector<bool> getD() {
        return std::vector<bool>({true,true,true,true,false});
    }
};

#endif
