/* Driver for the pendulum example
 *
 * Copyright (C) Michael Hanke 2019
 * Version 2020-06-09
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "nlPendel.hpp"
#include "RKCbasis.hpp"
#include "Legendre.hpp"
#include "Chebyshev.hpp"
#include "GaussLegendre.hpp"
#include "Xn.hpp"
#include "LSQSolver.hpp"
#include "DirectSolver.hpp"
#include "GnuPlot.hpp"
#include <Eigen/Dense>
#include <memory>
#include <iostream>
#include <fstream>
#include <cmath>
#include <vector>
#include <string>

using namespace Eigen;
using namespace std;
using namespace LSCM;

VectorXd inisol(const double t) {
    double p = 2.745, dy = 0.8284, dl = 5;
    double pfx = p/(2.0*M_PI);
    double pfy = pfx*0.5;
    double pfl = pfx;
    double ax = 2.0;
    double ay = 0.5*dy;
    double al = -0.5*dl;
    
    Eigen::VectorXd res(5);
    res(0) = ax*cos(pfx*t);
    res(1) = -ax*pfx*sin(pfx*t);
    res(2) = ay*cos(pfy*t)-3.0;
    res(3) = -ay*pfy*sin(pfy*t);
    res(4) = al*cos(pfl*t)+6.5;
    return res;
}

LSCMint main() {
    // Need a dae first
    shared_ptr<nlPendel> dae = make_shared<nlPendel>();
    vector<bool> D = dae->getD();

    bool prLSCMint = true;
    LSCMint n = 1;
    LSCMint N = 20;
    LSCMint M = N+1;
    double factol = 1e-3;
    shared_ptr<Grid> grid = make_shared<Grid>(A,B,n);
    DirectSolver lslv;
    
    cout << "*** Test of nonlinear pendel ***" << endl << endl;
    
    auto basis = make_shared<RKCbasis>(N);
    auto space = make_shared<Xn>(D,grid,basis);
    
    Yn::LSQnorm norm = Yn::LSQ_L2;
    shared_ptr<GaussLegendre> inttype = make_shared<GaussLegendre>(M);
    auto colloc = make_shared<Yn>(inttype,norm);
        
    GridFkt x0(space,inisol);
    LSQSolver slv(dae,space,colloc,lslv);
    slv.setAlpha(1.0);
    GridFkt sol = slv.nlsq(x0);
        
    auto basisL = make_shared<Legendre>(N);
    auto spaceL = make_shared<Xn>(D,grid,basisL);
    
    GridFkt x0L(spaceL,inisol);
    LSQSolver slvL(dae,spaceL,colloc,lslv);
    slvL.setAlpha(1.0);
    GridFkt solL = slvL.nlsq(x0L);

    auto basisC = make_shared<Chebyshev>(N);
    auto spaceC = make_shared<Xn>(D,grid,basisC);
    
    GridFkt x0C(spaceC,inisol);
    LSQSolver slvC(dae,spaceC,colloc,lslv);
    slvC.setAlpha(1.0);
    GridFkt solC = slvC.nlsq(x0C);
    
    // For demonstration purposes: Plot the solution
    GnuPlot gp1;
    vector<int> comp1 = {0,1};
    vector<string> label1 = {"x","y"};
    gp1.plot(solC,comp1,label1);
    gp1.show();
        
    GnuPlot gp2;
    vector<int> comp2 = {2,3};
    vector<string> label2 = {"u","v"};
    gp2.plot(solC,comp2,label2);
    gp2.show();
}
