/* File LinCaMo.hpp
 *
 * This is the definition of DAE1
 *
 * C Michael Hanke 2020
 * Version: 2020-05-13
 */

#ifndef LINCAMO_HPP
#define LINCAMO_HPP

#include "DAE.hpp"
#include <iostream>

using namespace std;
using namespace Eigen;

class LinCaMo : public LSCM::DAE {
private:
    double rho;
    double t0;
    VectorXd iv = VectorXd::Zero(4);
    
    VectorXd LinCaMo_ex()
    {
        VectorXd res(7);
        double st = sin(t0);
        double ct = cos(t0);
        res << st,
               ct,
               2.0*(ct*ct),
               ct,
              -st,
              -2.0*sin(2.0*t0),
              -st/rho;
        return res;
    }

public:
    LinCaMo(double rho, double t = 0.0) : rho(rho), t0(t) {
        l = 4;
        iv = getGa()*LinCaMo_ex();
    }
    
    MatrixXd A(double t) {
        MatrixXd res = MatrixXd::Identity(7,7);
        res(6,6) = 0.0;
        return res;
    }

    MatrixXd B(double t) {
        MatrixXd res = MatrixXd::Zero(7,7);
        double s = sin(t);
        double c = cos(t);
        
        Matrix<double,3,3> Afrac {
            { 0.0, 1.0,  -c},
            {-1.0, 0.0,  -s},
            { 0.0, 0.0, 0.0}
        };
        Matrix<double,3,1> Bfrac {
            {-c*c*2.0*rho}, {-s*c*2.0*rho}, {s*2.0*rho}
        };
        Matrix<double,3,3> Cfrac {
            {0.0, 0.0,   s},
            {0.0, 0.0,  -c},
            {0.0, 0.0, 1.0}
        };
        
        res.block(0,3,3,3) = -Matrix<double,3,3>::Identity();
        res.block(3,0,3,3) = Cfrac;
        res.block(3,3,3,3) = Afrac;
        res.block(3,6,3,1) = Bfrac;
        res.bottomLeftCorner(1,3) = -Bfrac.transpose();
        
        return res;
    }

    VectorXd q(double t) {
        VectorXd res(7);
        double st = sin(t);
        double ct = cos(t);
        res << 0.0,
               0.0,
               0.0,
               st*(8.0*(ct*ct)-2.0),
               ct*(6.0*(st*st)-2.0*(ct*ct)-2.0),
               -2.0*cos(2*t),
               0.0;
        return res;
    }
    
    vector<bool> getD() {
        return vector<bool>({true,true,true,true,true,true,false});
    }
    
    MatrixXd getGa() {
        double s = sin(t0);
        double c = cos(t0);
        double s2 = sin(2.0*t0);
        double c2 = cos(2.0*t0);
        
        Matrix<double,2,3> H {
            {  s,  -c, 0.0},
            {0.0, 1.0,   c}
        };
        Matrix<double,3,3> Afrac {
            { 0.0, 1.0,  -c},
            {-1.0, 0.0,  -s},
            { 0.0, 0.0, 0.0}
        };
        Matrix<double,3,1> Bfrac {
            {-c*c}, {-s*c}, {s}
        };
        Matrix<double,3,3> Omega = Bfrac*Bfrac.transpose();
        Matrix<double,3,1> dBfrac {
            {s2}, {-c2}, {c}
        };
        Matrix<double,3,3> dOmega = dBfrac*Bfrac.transpose()
                                    +Bfrac*dBfrac.transpose();
        
        MatrixXd res = MatrixXd::Zero(4,7);
        res.topLeftCorner(2,3) = H;
        res.bottomLeftCorner(2,3) = H*(Afrac+dOmega)*Omega;
        res.block(2,3,2,3) = H;
        
        return res;
    }
    
    MatrixXd getGb() {
        return MatrixXd::Zero(4,7);
    }
    
    VectorXd getr() {
        return iv;
    }
    
    void setIv(const VectorXd& ivn) { iv = ivn; }
    
};

#endif
