/* Test driver for the index-3 example DAE1
 * Check of p-version
 *
 * Copyright (C) Michael Hanke 2020
 * Version 2020-03-25
 */

/* 
    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.

*/

#include "DAE2.hpp"
#include "RKCbasis.hpp"
#include "Legendre.hpp"
#include "Chebyshev.hpp"
#include "GaussLegendre.hpp"
#include "Xn.hpp"
#include "LinLSQMatrices.hpp"
#include "LinLSQSolver.hpp"
#include "WeightedSolver.hpp"
#include "DirectSolver.hpp"
#include <Eigen/Dense>
#include <memory>
#include <iostream>
#include <cmath>

using namespace Eigen;
using namespace std;
using namespace LSCM;

#include "DAE2_exsol.cpp"

LSCMint main() {
    // parameters similar to the matlab implementation
    bool prLSCMint = true;
    LSCMint n = 50;
    LSCMint N = 5;
    LSCMint M = N+1;
    
    auto basis = make_shared<RKCbasis>(N);
    auto basisL = make_shared<Legendre>(N);
    auto basisC = make_shared<Chebyshev>(N);
    auto grid = make_shared<Grid>(0.0,1.0,n);
    
    Yn::LSQnorm norm = Yn::LSQ_L2;
    shared_ptr<GaussLegendre> inttype = make_shared<GaussLegendre>(M);
    auto colloc = make_shared<Yn>(inttype,norm);
    
    // Need a dae first: DAE2
    cout << "  *** Solving DAE2 ***" << endl << endl;
    auto dae2 = make_shared<DAE2>(-25.0,-1.0);
    vector<bool> D1 = dae2->getD();
    auto space1 = make_shared<Xn>(D1,grid,basis);
    
    auto linlsq1 = make_shared<LinLSQMatrices>(dae2,space1,colloc);
    WeightedSolver slv(linlsq1);
    LinLSQSolver& slvp = slv;
    
    GridFkt sol1 = slvp.solve();
    
    GridFkt exsol1(space1,DAE2ex);
    GridFkt error1 = sol1-exsol1;
    cout << "DAE solved!" << endl << "These are the errors:" << endl;
    cout << "L2 (comp): " << error1.cl2norm().transpose() << endl;
    cout << "Inf (comp): " << error1.cinfnorm().transpose() << endl;
    cout << "L2: " << error1.l2norm() << endl;
    cout << "H1: " << error1.h1norm() << endl;
    cout << "Using the exact solution:" << endl;
    cout << "L2 (comp): " << sol1.cl2dist(DAE2ex).transpose() << endl;
    cout << "Inf (comp): " << sol1.cinfdist(DAE2ex).transpose() << endl;
    cout << "L2: " << sol1.l2dist(DAE2ex) << endl;
    cout << "H1: " << sol1.h1dist(DAE2ex,DAE2dex) << endl;
    
    shared_ptr<Xn> spaceL1 = make_shared<Xn>(D1,grid,basisL);
    
    auto linlsqL1 = make_shared<LinLSQMatrices>(dae2,spaceL1,colloc);
    WeightedSolver slvL(linlsqL1);
    LinLSQSolver& slvLp = slvL;
    
    GridFkt solL1 = slvLp.solve();
    GridFkt diffL1 = sol1-solL1;
    
    cout << endl << "Now using Legendre polynomials" << endl;
    cout << "Errors:" << endl;
    cout << "L2 (comp): " << solL1.cl2dist(DAE2ex).transpose() << endl;
    cout << "Inf (comp): " << solL1.cinfdist(DAE2ex).transpose() << endl;
    cout << "L2: " << solL1.l2dist(DAE2ex) << endl;
    cout << "H1: " << solL1.h1dist(DAE2ex,DAE2dex) << endl;
    cout << "Diff to RK solutions" << endl;
    cout << "L2 (comp): " << diffL1.cl2norm().transpose() << endl;
    cout << "Inf (comp): " << diffL1.cinfnorm().transpose() << endl;
    cout << "L2: " << diffL1.l2norm() << endl;
    cout << "H1: " << diffL1.h1norm() << endl;

    shared_ptr<Xn> spaceC1 = make_shared<Xn>(D1,grid,basisC);
    
    auto linlsqC1 = make_shared<LinLSQMatrices>(dae2,spaceC1,colloc);
    slvL.replaceLSQMatrices(linlsqC1);
    
    GridFkt solC1 = slvLp.solve();
    GridFkt diffC1 = sol1-solC1;

    cout << endl << "Now using Chebyshev polynomials" << endl;
    cout << "Errors:" << endl;
    cout << "L2 (comp): " << solC1.cl2dist(DAE2ex).transpose() << endl;
    cout << "Inf (comp): " << solC1.cinfdist(DAE2ex).transpose() << endl;
    cout << "L2: " << solC1.l2dist(DAE2ex) << endl;
    cout << "H1: " << solC1.h1dist(DAE2ex,DAE2dex) << endl;
    cout << "Diff to RK solutions" << endl;
    cout << "L2 (comp): " << diffC1.cl2norm().transpose() << endl;
    cout << "Inf (comp): " << diffC1.cinfnorm().transpose() << endl;
    cout << "L2: " << diffC1.l2norm() << endl;
    cout << "H1: " << diffC1.h1norm() << endl;

}
