#  Computes matrices for scalar Helmholz equation
#
#     -u''(x)+k(x)*u(x)=λ u
#     u'(a)=0
#     u'(b)=mu *u(b)
#
#  Returned matrices: B=[B1,B2,B3] corresponding to the eigvalprob
#
#      (B1+λ*B2+mu*B3)u=0
#
#



# Solves same problem but with dtn on the right:
#     u'(b)=0
#     u'(a)=mu *u(0)
function helmholtz_fd_left_mu(a,b,m,kk)
    L=b-a;
    xv=range(a,stop=b,length=m);
    h=xv[2]-xv[1];
    α=-3/(2*h)
    β=2/h
    γ=-1/(2*h)
    kv=kk.(xv[2:end-1]);
    B1=diagm(0 => [α ; -2*ones(m-2)/h^2 + kv; -α],
             -1 => [ones(m-2)/h^2 ; -β],
             1 => [ β ; ones(m-2)/h^2 ])
    B1[1,3]=γ;
    B1[end,end-2]=-γ
    B2=diagm(0 => [0 ; -ones(m-2) ; 0])
    B3=zeros(m,m); B3[1,1]=-1;
    B1=sparse(B1);
    B2=sparse(B2);
    B3=sparse(B3);

    return (xv,[B1,B2,B3]);
end



function helmholtz_fd_old(a,b,m,kk)
    (xv,B)=helmholtz_fd_left_mu(-b,-a,m, x-> kk(-x));
    B1=reverse(reverse(B[1],dims=1),dims=2)
    B2=reverse(reverse(B[2],dims=1),dims=2)
    B3=reverse(reverse(B[3],dims=1),dims=2)
    return (-reverse(xv),[B1,B2,-B3])
end

function helmholtz_fd(a,b,m,kk)
    L=b-a;
    xv=range(a,stop=b,length=m);
    h=xv[2]-xv[1];
    α=-3/(2*h)
    β=2/h
    γ=-1/(2*h)
    kv=kk.(xv[2:end-1]);
    B1=spdiagm(0 => [1 ; -2*ones(m-2)/h^2 + kv; -α],
             -1 => [ones(m-2)/h^2 ; -β],
             1 => [ 0 ; ones(m-2)/h^2 ])
    B1[end,end-2]=-γ
    B2=spdiagm(0 => [0 ; -ones(m-2) ; 0])
    B3=spzeros(m,m); B3[end,end]=-1.0;
    return (xv,[B1,B2,B3]);
end
