#  Computes matrices for scalar Helmholtz equation
#
#     -u''(x)+k(x)*u(x)=λ u
#     u'(a)=mu*u'(a)
#     u'(b)=0
#
#  using a spectral discretization.
#
#  Returned matrices: B=[B1,B2,B3] corresponding to the eigvalprob
#
#      (B1+λ*B2+mu*B3)u=0
#
#


# Scaled chebyshev poly discretization
function cheb(N,a=-1,b=1)
    if N==0
        D=0;
        x=1;
        return (D,x)
    end
    x = cos.(pi*(0:N)/N);
    c = [2 ; ones(N-1) ; 2].*(-1).^(0:N)
    X = repeat(x,1,N+1)
    dX = X-X';
    II=Matrix{Float64}(I,N+1,N+1)
    D  = (c*(1 ./ c)') ./ (dX+II);
    D  = D - diagm(0 => D*ones(N+1));

    D=D*2/(b-a);
    x=(x .+ 1)*(b-a)/2 .+ a;

    return (D,x)
end


function cheb(n,x)
    return cos.(n*acos.(x));
end



function cheb_eval(xv,c,a,b)
    V=zeros(eltype(xv),size(xv))
    for k=1:size(c,1)
        V += c[k]*cheb(k-1,2*xv/(b-a).-1);
    end
    return V
end




function helmholtz_cheb(a,b,m,kk)

    L=b-a;
    (D,x)=cheb(m-1,a,b)
    D2=D^2+diagm(0=>kk.(x))
    D2[1,:]=D[1,:];
    D2[m,:]=D[m,:];

    B1=D2;
    B3=zeros(m,m);
    B3[m,:]=-[zeros(m-1);1];
    II2=diagm(0=> [0;ones(m-2);0]);
    B2=-II2
    return (x,[B1,B2,B3]);

end
