# Creator functions for the nonlinearized two-param evp.

function gder(B, l;der=0,target=0,eigval_number=1)
    T=ComplexF64
    BB=(l,s) -> B[1]+l*B[2]+s*B[3];
    m=size(B[1],1);

    Z=eigen(Matrix(B[1]+l*B[2]),-Matrix(B[3]));
    vals=Z.values;


    vals[isnan.(vals)] .= Inf; # Remove infinite eigenvalues

    # Find the index of eigval_number, when vals sorted by distance to target
    k=sortperm(abs.(vals .- target))[eigval_number];

    c=ones(T,m); # Hardcoded c-vector
    p=der;
    Y=zeros(T,m,p+1);

    # Set initial vectors
    sv=zeros(T,der+1);
    sv[1]=vals[k];
    s=sv[1];  # sv[k+1] contains derivative k

    y=Z.vectors[:,k] / (c'*Z.vectors[:,k]); # don't forget to normalize by c
    Y[:,1]=y;

    # Matrix in linear system for derivative computations
    C=[BB(l,s) B[3]*y ; c' 0];
    CF=C
    if (p>2) # Only LU-factorize if we need many derivatives
        CF=factorize(C);
    end


    for i=1:p
        # Compute derivative i:


        # Right-hand side
        b=zeros(T,m+1);
        for k=1:i-1
            b[1:m] += -binomial(i,k)*(B[3]*Y[:,1+k])*sv[1+i-k]
        end
        # One extra term for k=i-1
        b[1:m] += -binomial(i,i-1)*B[2]*Y[:,1+i-1];

        # Solve it
        yy=CF\b;
        Y[:,i+1]=yy[1:m];
        sv[i+1]=yy[end];
    end

    return (sv,y)
end


function twoparam_Mder(A,B,target,l,der,bigfloat=false,eigval_number=1)
    if (der==0)
        if (bigfloat)
            sv=gder_bigfloat(B,l,der=0,eigval_number=eigval_number)[1];
        else
            sv=gder(B,l,der=0,target=target,eigval_number=eigval_number)[1];
        end
        # Sometimes sv[1] becomes Inf. Then we get out of mem error
        M1=A[1]+l*A[2];
        M2=sv[1]*A[3];
        M=M1+M2
    else
        if (bigfloat)
            sv=gder_bigfloat(B,l,der=der,eigval_number=eigval_number)[1];
        else
            sv=gder(B,l,der=der,target=target,eigval_number=eigval_number)[1];
        end
        M=sv[der+1]*A[3];
        if (der==1)
            M+=A[2];
        end
    end
    return M
end

function twoparam_Mlincomb(A,B,target,l,V,bigfloat=false,eigval_number=1)
    T=ComplexF64;
    n=size(A[1],1);
    q=size(V,2);
    sv=gder(B,l,der=q-1,target=target,eigval_number=eigval_number)[1];
    z=zeros(T,n);
    for der=0:(q-1)
        if (der==0)
            z += A[1]*V[:,1]+l*A[2]*V[:,1];
        elseif (der==1)
            z += A[2]*V[:,2];
        end
        z += (A[3]*V[:,der+1])*sv[der+1];
    end
    return z;
end



# Modes: :spmf_taylor, :spmf_taylor_DerSPMF, :spmf_gder, :Mder
function TwoNEP(A,B;mode=:spmf_gder,σ=0,eigval_number=1,maxder=60,bigfloat=false)
    T=ComplexF64
    p=maxder
    if (mode==:spmf_taylor || mode == :spmf_taylor_DerSPMF)
        FF=zeros(T,p,3)
        FF[1,1]=1;
        FF[1,2]=σ
        FF[2,2]=1
        FF[:,3]=gder(B,σ,der=p-1,eigval_number=eigval_number)[1]

        f1=S->one(S);
        f2=S->S

        function f3_taylor(S)
            F=zero(S);
            for k=0:(p-1)
                F += ((S-one(S)*σ)^k)*FF[k+1,3]/factorial(float(k));
            end
            return F
        end
        f3=f3_taylor;
        spmf=SPMF_NEP(A,[f1 , f2 ,  f3],check_consistency=false);
        if (mode == :spmf_taylor)
            return spmf;
        else
            dnep=DerSPMF(spmf,σ,FF); # Constructor free creation
            return dnep
        end


    elseif (mode == :spmf_gder)
        f1=S->one(S);
        f2=S->S
        # Matrix function f3 a bit complicated. We only support scalars in this case.
        function f3_gder(S)
            if (S isa Number)
                return gder(B,S,eigval_number=eigval_number)[1][1]
            else
                return reshape([gder(B,S[1,1],eigval_number=eigval_number)[1][1]],size(S,1),size(S,2))
            end
        end
        f3=f3_gder
        spmf=SPMF_NEP(A,[f1 , f2 ,  f3],check_consistency=false);
        return spmf;
    else  # Mder. This can be slow (but easier to generalize to BigFloat)

        Mder=(l,der)  -> twoparam_Mder(A,B,σ,l,der,bigfloat,eigval_number);
        Mlincomb=(l,V)  -> twoparam_Mlincomb(A,B,σ,l,V,bigfloat,eigval_number);
        #nep=Mder_NEP(size(A[1],1),Mder);
        nep=Mder_Mlincomb_NEP(size(A[1],1),Mder,Mlincomb);

        return nep;
    end
end
