import Base.size;


"""
   type PeriodicDDE_NEP <: NEP

This type represents NEP associated with the time-periodic delay-differential equation
```math
\dot{x}(t)=A(t)x(t)+B(t)x(t-\tau)
```
where `A(t)` and `B(t)` are periodic functions with period `\tau`.

# Example
```julia-repl
julia> nep=PeriodictDDE_NEP((t) -> 3, (t) -> 4,1)
julia> λ,v=newton(nep,λ=3)
julia> compute_Mlincomb(nep,λ,v)
1-element Array{Complex{Float64},1}:
 0.0+0.0im
```
"""
abstract type PeriodicDDE_NEP <: NEP; end; 
type PeriodicDDE_NEP_DAE <: PeriodicDDE_NEP
    A::Function
    B::Function
    n::Integer
    N::Integer
    tau::Number;
    disconts::Vector{Float64};
    E::Matrix;
    isconst::Bool
    function PeriodicDDE_NEP_DAE(A::Function,B::Function,E::AbstractMatrix,tau::Number)
        n=size(A(0),1)
        nep=new(A,B,n,1000,tau,[],E,false);;
        return nep;
    end        
end

    
type PeriodicDDE_NEP_ODE <: PeriodicDDE_NEP
    A::Function
    B::Function
    n::Integer
    N::Integer
    tau::Number;
    disconts::Vector{Float64};
    #ode_solver::Function
    E::Matrix
    function PeriodicDDE_NEP_ODE(A::Function,B::Function,tau::Number)
        n=size(A(0),1)
        nep=new(A,B,n,1000,tau,[]);;
        #nep.disconts=detect_disconts(nep);
        return nep;
    end    
end



type NEP_Mder <: NEP
    Mder::Function
    n::Number
end
function compute_Mder(nep::NEP_Mder,λ::Number,der::Integer=0)
    print("Computing Mder");
    MM=nep.Mder(λ);
    println(".");
    return MM
end

function size(nep::NEP_Mder,dim=-1)
    n=nep.n;
    if (dim==-1)
        return (n,n)
    else
        return n
    end
end


function size(nep::PeriodicDDE_NEP,dim=-1)
    n=nep.n;
    if (dim==-1)
        return (n,n)
    else
        return n
    end
end


function myode(f,a,b,N,y0)
    h=(b-a)/N
    t=a;
    yy=copy(y0);
    for k=1:N
        yy=yy+h*f(t,yy)
        t=t+h;
    end
    return yy;
end

# Stand-alone implementation of RK4
function ode_rk4(f,a,b,N,y0)
    h=(b-a)/N; t=a;
    y=copy(y0);
    for k=1:N
        s1=h*f(t,y);
        s2=h*f(t+h/2,y+s1/2);
        s3=h*f(t+h/2,y+s2/2);
        s4=h*f(t+h,y+s3);
        y[:]=y+(s1+2*s2+2*s3+s4)/6;
        t=t+h;
    end
    return y;
end

function ode_be_dae(A,E,a,b,N,y0)
    h=(b-a)/N;
    y=copy(y0);
    t=a+h;
    for k=1:N
        y=(h*A(t)-E)\(E*y);
        t=t+h;
    end
    return y;
end

#function ode_be_dae_const(A,E,a,b,N,y0)
#    h=(b-a)/N;
#    y=copy(y0);
#    t=a+h;
#    AA=A(t);
#    Afact=factorize(h*AA-E);
#    for k=1:N
#        y=Afact\(E*y);
#        t=t+h;
#    end
#    return y;
#end
#
#abstract type ODESolver end;
#type RK4Solver <: ODESolver; end
#
#function forward_solve(nep::PeriodicDDE_NEP,S,V,Type{RK4Solver})
#
#    h=(b-a)/N; t=a;
#    y=copy(y0);
#    for k=1:N
#        s1=h*f(t,y);
#        s2=h*f(t+h/2,y+s1/2);
#        s3=h*f(t+h/2,y+s2/2);
#        s4=h*f(t+h,y+s3);
#        y=y+(s1+2*s2+2*s3+s4)/6;
#        t=t+h;
#    end
#    return y;
#    
#end
#
    

#function compute_Mlincomb(nep::PeriodicDDE_NEP, λ::Number, v::Any)
#    n=size(nep,1);
#    # We are using (non-trivial) fact that
#    # the MM satisfies an ODE (as well as the action)
#    F=(t,Y) -> (nep.A(t)*Y+nep.B(t)*Y*exp(-(nep.tau*λ))-Y*λ)
#    Y0=v;
#    YY=ode_rk4(F, 0,nep.tau, nep.N, Y0);
#    #YY=myode(F, 0,nep.tau, nep.N, Y0);    
#    return YY-Y0
#end
#

function compute_MM(nep::PeriodicDDE_NEP_DAE, S ,V)
    if (size(V,2)>1)
        println(size(V));
        error("Not implemented");
    end
    Af= t -> (nep.A(t)+nep.B(t)*exp(-nep.tau*S[1,1])-S[1,1]*nep.E);
    Y0=V;
    if (!nep.isconst)
        YY=ode_be_dae(Af,nep.E, 0,nep.tau, nep.N, Y0);
        return YY-Y0;
    else
        YY=ode_be_dae_const(Af,nep.E, 0,nep.tau, nep.N, Y0);
        return YY-Y0;
    end
    
end


### The implementations of compute_MM is a 
### crude first-version, not optimized for efficiency nor accuracy.
function compute_MM(nep::PeriodicDDE_NEP_ODE, S ,V)
    n=size(nep,1);
    # We are using (non-trivial) fact that
    # the MM satisfies an ODE (as well as the action)
    local F::Function;
    if size(S,1)==1
        F=(t,Y) -> (nep.A(t)*Y+nep.B(t)*Y*exp(-nep.tau*S[1,1])-Y*S[1,1])
    else
        F=(t,Y) -> (nep.A(t)*Y+nep.B(t)*Y*expm(-full(nep.tau*S))-Y*S)
    end

    
    Y0=V; # Use RK4 for ODE's
    YY=ode_rk4(F, 0,nep.tau, nep.N, Y0);
    #YY=myode(F, 0,nep.tau, nep.N, Y0);    
    return YY-Y0
end

function compute_Mlincomb(nep::PeriodicDDE_NEP,λ,v)

    return compute_MM(nep,[λ],v);
end


##
#function compute_MM(nep::PeriodicDDE_NEP, S ,V)
#    n=size(nep,1);
#    # We are using (non-trivial) fact that
#    # the MM satisfies an ODE (as well as the action)
#    F=(t,Y) -> (nep.A(t)*Y+nep.B(t)*Y*expm(-full(nep.tau*S))-Y*S)
#    Y0=V;
#    t0=0;
#    disconts_and_tau=[nep.disconts;nep.tau];
#    YY=copy(V);
#    for k=1:size(disconts_and_tau,1)
#        YY=ode_rk4(F, t0,disconts_and_tau[k], nep.N, Y0);
#        Y0=YY;
#        t0=disconts_and_tau[k];
#    end
#    #YY=myode(F, 0,nep.tau, nep.N, Y0);    
#    return YY-V
#end


# For compute_Mlincomb we (implicitly) use compute_Mlincomb_from_MM 

function compute_Mder(nep::PeriodicDDE_NEP,λ::Number,der::Integer=0)
    if (der==0)
        Z=zeros(typeof(λ),size(nep,1),size(nep,1));
        for k=1:size(nep,1)
            if (mod(k,20)==1)
                println("k=",k);
            end
            
            ek=copy(Z[:,k]); ek[k]=1;
            Z[:,k]=compute_Mlincomb(nep,λ,ek);
        end
        return Z;
        #return compute_Mder_from_MM(nep,λ,der) 
    elseif (der==1)
        # Compute first derivative with finite difference. (Slow and inaccurate)
        ee=sqrt(eps())/10;
        Yp=compute_Mder(nep,λ+ee,0)
        Ym=compute_Mder(nep,λ-ee,0)
        return (Yp-Ym)/(2ee);
    else # This would be too slow.
        error("Higher derivatives not implemented");
    end

end



"""
    nep=problem_gallery(PeriodicDDE_NEP; name)

Constructs a PeriodicDDE object from a gallery. The default is "mathieu", which is
the delayed Mathieu equation in section 1 in reference.

Subset of eigenvalues of the default example:
 -0.24470143590830754
 -0.561610418452567 - 1.511169478595549im
 -0.561610418452567 + 1.511169478595549im
 -1.859617846506182 - 1.261010754174415im
 -1.859617846506182 + 1.261010754174415im
 -2.400415351524992 - 1.323058902477801im
 -2.400415351524992 + 1.323058902477801im


# Example
```julia-repl
julia> nep=nep_gallery(PeriodicDDE_NEP,name="mathieu")
julia> (λ,v)=newton(Float64,nep,λ=-0.2,v=[1;1])
(-0.24470143590830754, [-1.25527, 0.313456])
julia> exp(nep.tau*λ)  # Reported in Figure 2 with multipliers in reference
0.6129923199095035
```
# Reference
* E. Bueler, Error Bounds for Approximate Eigenvalues of Periodic-Coefficient Linear Delay Differential Equations, SIAM J. Numer. Anal., 45(6), 2510–2536

"""
function problem_gallery(::Type{PeriodicDDE_NEP}; name::String="mathieu",n=200,TT=Complex128)
    if (name == "mathieu")
        δ=1; b=1/2; a=0.1; tau=2;
        A=t-> [0 1; -( δ+ a*cos(pi*t) ) -1];
        B=t-> [0 0; b 0];
        nep=PeriodicDDE_NEP(A,B,tau)
        return nep;
    elseif (name == "rand0")
        # Some eigenvalues for n=200: 
        #  4.63633+1.10239im
        # 4.63633-1.10239im
        # 5.58214+4.03225im
        # 5.58214-4.03225im
        # 5.73989+0.732386im
        # 5.73989-0.732386im

        srand(0);
        A0=sprandn(n,n,0.3)-speye(n,n)
        A1=sprandn(n,n,0.3)-speye(n,n)
        B0=sprandn(n,n,0.3)-speye(n,n);
        B1=sprandn(n,n,0.3)-speye(n,n);            
        tau=2;
        A=t-> A0+cos(pi*t)*A1;
        B=t-> B0+exp(0.01*sin(pi*t))*B1;
        nep=PeriodicDDE_NEP(A,B,tau)
        return nep;
    elseif (name == "discont")
        δ=1; b=1/2; a=0.1; tau=2;
        A=t-> [0 1; -( δ+ a*cos(pi*t) ) -1]+eye(2)*((t-0.3)^2)*(t>0.3);
        B=t-> [0 0; b 0];
        nep=PeriodicDDE_NEP(A,B,tau)
    elseif (name == "milling0")
        δ=1; b=1/2; a=0.1; tau=2;
        A0=t-> [0 1; -( δ+ a*cos(pi*t) ) -1]+eye(2)*((t-0.3)^2)*(t>0.3);
        B0=t-> [0 0; b 0];
        e=ones(n-1);
        DD=spdiagm((e,-2*e,e),(-1,0,1));
        DD[1,1]=-1;
        h=1/n;
        DD=DD
        a=1;
        A=t-> [sparse(A0(t)) a*speye(2,n); a*speye(n,2) DD];
        B=t-> [sparse(B0(t)) spzeros(2,n); spzeros(n,2) -0*speye(n)];        
        #DD=full(spdiagm((e,-2*e,e),(-1,0,1)));
        #A=t-> [(A0(t)) eye(2,n); eye(n,2) DD];
        #B=t-> [(B0(t)) zeros(2,n); zeros(n,2) eye(n)];        

        nep=PeriodicDDE_NEP(A,B,1)
    elseif (name == "milling1_be")
        # The milling model in analyzed by Insperger, Orosz, Bueler, etc (with unit constants)
        omega0=TT(1);
        zeta0=TT(1);  # Zeta (damping)
        m  =TT(1);      # Mass 
        ap =TT(1)
        KR =TT(1)
        KT =TT(1);
        tau=TT(1)

        A0=[TT(0) TT(1); -omega0^2 -2*zeta0*omega0];

        phi=t -> 2*pi*t/tau;
        
        h=t -> (t<real(tau)/2).*(sin(phi(t)).^2*KR+KT*cos(phi(t)).*sin(phi(t)));
        E21=zeros(TT,2,2); E21[2,1]=1;

        # We need to use implicit method (default for DAEs).
        nep=PeriodicDDE_NEP_DAE(t-> A0-E21*h(t)*ap/m,
                                t->+E21*h(t)*ap/m,eye(TT,2),1)

        nep.N=50;
        return nep;
    elseif (name == "milling1_rk4")
        # The milling model in analyzed by Insperger, Orosz, Bueler, etc (with unit constants)
        omega0=TT(1);
        zeta0=TT(1);  # Zeta (damping)
        m  =TT(1);      # Mass 
        ap =TT(1)
        KR =TT(1)
        KT =TT(1);
        tau=TT(1)

        A0=[TT(0) TT(1); -omega0^2 -2*zeta0*omega0];

        phi=t -> 2*pi*t/tau;
        
        h=t -> (t<tau/2).*(sin(phi(t)).^2*KR+KT*cos(phi(t)).*sin(phi(t)));
        E21=zeros(TT,2,2); E21[2,1]=1;

        # We need to use implicit method (default for DAEs).
        nep=PeriodicDDE_NEP_ODE(t-> A0-E21*h(t)*ap/m,
                                t->+E21*h(t)*ap/m,1)
        nep.N=50;
        return nep;
        
        
        
    elseif (name == "milling")            
        # The problem by Rott and Hömberg (and Jarlebring)
        #error("Problem in http://dx.doi.org/10.3182/20100607-3-CZ-4010.00023 not yet implemented")
        m=setDefaultParameters(n); m=computeMatricesScaled(m);
        ap=0.1;nn=3200;

        #x=1e-2;d=[ones(m.disc_n0+1);x*ones(m.disc_n0+1)];
        #D=eye(2*m.disc_n0+2);
        

        A=t-> -AScaled(t,nn/60,ap,m)/100;
        B=t-> -BScaled(t,nn/60,ap,m)/100;
        nep=PeriodicDDE_NEP(A,B,1)
        return nep;
    elseif (name == "newmilling")


    else
        error("Unknown PeriodicDDE_NEP type:",name);
    end
end




type MillingParams
    ####################################
    #werkzeug-ode
    ode_m;
    #val.ode.m;
    ode_f0;
    ode_zeta;
    #val.ode.zeta;
    ####################################
    #werkstueck
    pde_beta;
    pde_p0;
    pde_E;
    pde_L;
    pde_A;
    pde_hist;
    ####################################
    #Prozess
    proc_phiSt;
    proc_phiEx;
    proc_Kt;
    #val.proc.Kt;
    proc_Kr;
    #val.proc.Kr;
    proc_xf;
    proc_zahn;
    ####################################
    #diskretisierung und solver
    disc_interv;
    disc_n0;
    disc_RelTol;
    disc_AbsTol;
    ####################################
    #coupling
    coupling;
    ####################################
    #stability lobe diagram parameters
    sld_tol;
    sld_RI_thr;
    sld_apVar;
    sld_N0;
    sld_NE;
    sld_Nincr;
    ####################################
    #parameter fuer die eigenwertanalyse
    evp_dim;
    evp_NcollPts;
    evp_gamma;
    mat_Z;
    mat_Agi;
    mat_C1;
    mat_C2;    
    mat_I;
    mat_D1;
    mat_B2;
    mat_A1;
    mat_A2;
    mat_M;
    mat_M2;
    ap0;
    function MillingParams()
        return new();

    end
    
end

#
function setDefaultParameters(nn)
    m=MillingParams()
    
    ####################################
    #werkzeug-ode
    m.ode_m=2.573;                   #mass ex. Butcher
    #val.ode.m=0.06;                     #mass ex. Oliver
    m.ode_f0=1.464257307328339e+02;  #frequency ex. Butcher
    m.ode_zeta=0.0032;               #Damping ex. Butcher
    #val.ode.zeta=0.012;                 #Damping ex. Oliver
    ####################################
    #werkstueck
    m.pde_beta=2e-7;                  #Material damping (Voigt-Model)
    m.pde_p0=2.7e-1;                  #density
    m.pde_E=70e4;                     #Elastizitaetsmodul
    m.pde_L=0.1;                      #Laenge des Balkens
    m.pde_A=10;                       #Flaeche des Balkens in mm
    m.pde_hist=-1e-4;                 #initial condition const. for td-solver
    ####################################
    #Prozess
    m.proc_phiSt=0.0;                 #Eintrittswinkel des Fraesers
    m.proc_phiEx=58.26*pi/180;                  #Austrittswinkel des Fraesers
    m.proc_Kt=5.5e2;             # Cutting const ex. Butcher in N/mm^(1+xf)
    #val.proc.Kt=894.0;                  # Cutting const for model Oliver
    m.proc_Kr=2.0e2;             # Cutting const ex. Butcher in N/mm^(1+xf)
    #val.proc.Kr=690.0;                  # Cutting const for model Oliver
    m.proc_xf=1.0;
    m.proc_zahn=1;                    #Anzahl der Zaehne des Fraesers
    ####################################
    #diskretisierung und solver
    m.disc_interv=100;                #Anzahl der Intervalle in denen eine ode geloest wird
    m.disc_n0=nn;                      #n0+1 Ortsdiskritisierungspunkte
    m.disc_RelTol=1e-3;               #Relative Toleranz des ODE-Loesers
    m.disc_AbsTol=1e-6;               #Absolute Toleranz des ODE-Loesers
    ####################################
    #coupling
    m.coupling=[1 1;1 1]; # 1,1-pde, 2-2 ode
    ####################################
    #stability lobe diagram parameters
    m.sld_tol=0.01;
    m.sld_RI_thr=0.;
    m.sld_apVar=0.1;
    m.sld_N0=3000;                    #starting spindle speed for stability lobe diagram
    m.sld_NE=4000;                   #final spindle speed for stability lobe diagram
    m.sld_Nincr=10.0;                #spindle speed increase
    #val.sld.outfile='karteBenchI';
    #val.sld.outfile='karteBenchII';
    ####################################
    #parameter fuer die eigenwertanalyse
    m.evp_dim=2*(m.coupling[1,1]*m.disc_n0+m.coupling[2,2]); #problem dimension
    #m.evp_NcollPts=80;               #number of collocation points for ex. Butcher
    m.evp_gamma=NaN;                  # scaling factor for the time variable
    #phi=linspace(0.01,pi-0.01,5);              # grid to find the initial eigenvalues
    #val.evp.guessLv0=complex(-0.02,phi(2:4)); #---"--- (general case)
    #val.evp.guessLv0=-0.007370+1.675017i;#value for benchI
    #val.evp.guessLv0=-0.315511+2.486569i;#value for benchII, par2=10100, par1 =1.5
    #val.evp.guessLv0= -0.073428+2.817029i;#value for benchIII, par2=10300, par1 =0.5
    #val.evp.bounds=[0.2 3.];           # bounds for the imaginary part of 
                                        # the followd eigenvalue


    m.ap0=0.5;nn=3200;
    m.evp_gamma=(nn/60)*(m.proc_zahn);
    return m;
end






function AScaled(t,n,ap,par::MillingParams)        
    h1Val= h1(t,n,par);        # Auswertung der Vergangenheitsfunktion h1
    Bg=[(par.mat_C2-ap*h1Val*par.mat_D1) par.mat_B2 ; par.mat_Z -par.mat_I];

    val=par.mat_Agi*Bg;
    return val;
end



function BScaled(t,n,ap,par::MillingParams)
    h1Val= h1(t,n,par);        # Auswertung des per. Koeffizienten
    Dg=[ap*h1Val*par.mat_D1 par.mat_Z;par.mat_Z par.mat_Z];
    val=par.mat_Agi*Dg;
    return val;
end

function h1(tbar,n,par::MillingParams)
    Kt=par.proc_Kt;
    Kr=par.proc_Kr;
    xf=par.proc_xf;

    t=tbar/par.evp_gamma;
    om=2.*pi*n;

    local out;
    for i=0:(par.proc_zahn-1)
        win=om*t+i*(2.0*pi/par.proc_zahn);
        Cwin=cos(win);
        Swin=sin(win);
        out1=g(win,par.proc_phiSt,par.proc_phiEx)*(Swin)^(xf)*(Kt*Cwin+Kr*Swin);
        out=1000*out1;  #weil kraft in milli Newton
    end
    return out;
end

function g(win,einwink,auswink)
    #sogenannte Schalterfunktion welche 1 ist wenn der Fraeser mit dem Material
    #in Kontakt kommt und Null falls der Fraeszahn "frei" haengt
    psi=atan((sin(einwink)-sin(auswink))/(cos(einwink)-cos(auswink)));
    keineausloesch=2*cos((2*pi+win+einwink-2*psi)/2)*sin((2*pi+win-einwink)/2);
    out=0.5*(1+sign(keineausloesch));
    return out;
end




function computeMatricesScaled(par::MillingParams)
        #initialize return value
        out=par;
        n0=par.disc_n0;

    #check type of problem
        modus=1;
 
    ##############################
        #gekoppeltes problem
        ##############################
       if modus==1
        #initialize return value
            h=par.pde_L/n0;
           b0=par.pde_beta*par.pde_E*par.pde_p0;   #Reibung
           p=zeros(n0-1);
            p[1:(n0-1)]=1;    #Vektor aus dem R^(n0-1) mit allen Komponenten gleich 1
            p1=(1/6)*h*p;       #obere und untere Nebendiagonale von A

            p2=[(2/3)*h*p;(1/3)*h];       #Hauptdiagonale
            #p2[n0]=(1/3)*h;   #    - " -

        #####################################################
        #berechnt einen Einheits-Matrix und eine Null-Matrix#
        #####################################################
            Z=zeros(n0+1,n0+1);
            out.mat_Z=Z;
            I=eye(n0+1,n0+1);  #erzeugt eine Einheitsmatrix der groesse n0+1 mal n0+1
            out.mat_I=I;

        ###################################################
        #berechne Massenmatrix A1, A2 und die Inverse von #
        #          0  A2
        #          1  0
        ###################################################
            v1=zeros(n0);
            v1[1:n0]=0;
            A1=diagm(p2,0)+diagm(p1,-1)+diagm(p1,+1);  #Erzeugt Tridiagonalmatrix mit p1 auf den Nebendiagonalen und p2 auf der Diagonalen
            out.mat_A1=A1;
            A2 = [A1 v1; v1' 1];
            A2 = (par.evp_gamma)*(par.evp_gamma) * A2;
            Z=zeros(n0+1,n0+1);
            Ag=[Z A2;I Z];                              #Gesamtsystemmatrix zum ODE System 1. Ordnung
            out.mat_Agi=inv(Ag);                        #inverse der Gesamtsystemmatrix des Systems 1.Ordnung
            out.mat_M=A1;                               #in M wird die Massenmatrix des Systems 2. Ordnung gespeichert
            out.mat_M2=A2;                              #
        ###########################################
        #erzeugt die Matrix B1,B2(Daempfungsmatrix)#
        ###########################################

            q1=-(1/h).*p;                           #obere und untere Nebendiagonale von B
            q2=[(2/h).*p;1/h];                            #Hauptdiagonale
            B1=diagm(q1,-1)+diagm(q1,+1)+diagm(q2,0);
            
            B2 = [(b0/par.pde_p0).*B1 v1;v1' 2*par.ode_zeta*(2.0*pi*par.ode_f0)];
            out.mat_B2 = (par.evp_gamma) * B2;
           
        #######################################
        #berechnet C1,C2 (Steifigkeitsmatrix)#
        #######################################
            r1=-(1./h)*p;                            #obere und untere Nebendiagonale von C
            r2=[(2./h)*p;1/h];                             #Hauptdiagonale
            C1=diagm(r1,-1)+diagm(r1,+1)+diagm(r2,0);
    
            v3=zeros(n0);
            v4=zeros(n0+1);
            v4[n0+1]=(2.0*pi*par.ode_f0)^2;
            out.mat_C2=[(par.pde_E/par.pde_p0).*C1 v3;v4'];

            out.mat_C1=C1;
        ###############################
        #berechne D1(Kopplungsmatrix) #
        ###############################

        

            out.mat_D1=zeros(n0+1,n0+1);
            out.mat_D1[n0,n0]=par.coupling[1,1]/(par.pde_A*par.pde_p0);
            out.mat_D1[n0+1,n0+1]=-par.coupling[2,2]/par.ode_m;
            out.mat_D1[n0+1,n0]=par.coupling[2,1]/par.ode_m;
            out.mat_D1[n0,n0+1]=-par.coupling[1,2]/(par.pde_A*par.pde_p0);
       end

     return out;

end
#


function detect_disconts(nep::PeriodicDDE_NEP;threshold=1e2,N=1000)
    xv=linspace(0,nep.tau,N)
    A=nep.A(xv[2])
    B=nep.B(xv[2])
    disconts=[];


    
    for k=3:size(xv,1)
        Anew=nep.A(xv[k])
        Bnew=nep.B(xv[k])
        A_ind=(norm(Anew-A,1)/mean([norm(A,1),norm(Anew,1)]))/abs(xv[k-1]-xv[k]);
        B_ind=(norm(Bnew-B,1)/mean([norm(B,1),norm(Bnew,1)]))/abs(xv[k-1]-xv[k]);
        if (((A_ind > threshold) && (norm(A,1)>=0) && (norm(Bnew,1)>=0))  ||
            ((B_ind > threshold) && (norm(B,1)>=0) && (norm(Bnew,1)>=0)) )
            disconts=[disconts;mean([xv[k],xv[k-1]])];
#            println("k=",k," ",A_ind, " ", B_ind)
        end
        B=Bnew;
        A=Anew;
    end
    return disconts;
                      
    

    

end


