/*
+-----------------------------------------------------------------------+
| BNS computes attractors in the synchronous Boolean network model      |
| v1.3 - February 2017                                                  |
|                                                                       |
| Maxim Teslenko                                                        |
| maximt@kth.se                                                         |
| https://people.kth.se/~dubrova/bns.html                               |
| KTH, Stockholm, Sweden                                                |
+-----------------------------------------------------------------------+

To compile assuming that minisat is in ../MiniSat_v1.14/ folder:
  g++ ../MiniSat_v1.14/Solver.o -I../MiniSat_v1.14 bns.c -o bns 
*/

#include "Solver.h"
#include <vector>


#define INDEX_OF_FIRST_VARIABLE 1
#define MAXIMUM_NUMBER_OF_RELEVANT_NODES_COUNTED 100
#define PRINT_STATE
#define PRINTF(X,Y)
#define PRINT(X) printf(#X":%d\n",X) 
//#define ASSERT_ON

#ifdef ASSERT_ON
#define ASSERT(x)           assert(x)  
#else
#define ASSERT(x)
#endif 


  /* Reading from file informaition*/
  int number_of_var,redunt_number_of_var,number_print_var;
  int *var2red_var;
  int *redunt_var2var;

  // Read from command line; 0 means no restrction is set and atractors of all length will be seacherd 
  int exact_atractor_size=0;

  int init_unfolding=0;

#define MAX_LENGTH 10000
#define MAX_NUMBER_OF_INPUTS 20
   /*---------------------------------*/


static struct {
  std::vector< std::vector<Lit> > *orig_clauses;
  Solver *S;
  unsigned number_of_var;
  unsigned depth;
}global_var;



#define toInt(x) index(x)


static unsigned        size_nonred_array;


void                   ReadNET(char *fileName,Solver& S);

/*Compare stats a and b in vector of stats stored in stats_sequance.*/
/*The first state in sequance assumed to have index 0, each state consists of sequence of values of */
bool compare_states(unsigned a,unsigned b,unsigned number_var, vec<lbool>& stats_sequance ){
  lbool *a_pr, *b_pr;
  unsigned size=stats_sequance.size();
  ASSERT(!( size < (a+1)*number_var || size < (b+1)*number_var ));

  a_pr=&stats_sequance[a*number_var];
  b_pr=&stats_sequance[b*number_var];

  while(number_var--){
    if(*a_pr != *b_pr) return false;
    a_pr++;
    b_pr++;
  }
  return true;

}

/*Make state with index a from the stats_sequance not allowed in future solutions of S in position with index b */
void constrain_state(unsigned a, vec<lbool>& stats_sequance, unsigned b, Solver &S, unsigned number_var ){

  vec<Lit> lits;
  int a_tmp=a*number_var;
  int b_tmp=b*number_var;

  ASSERT( a_tmp+number_var <= stats_sequance.size() );
  ASSERT( b_tmp+number_var <= S.nVars() );


    while(number_var--){
      if (stats_sequance[a_tmp] != l_Undef){

#ifdef PRINT_STATE
	//printf( "%s", (stats_sequance[a_tmp]==l_True)?"1":"0");
#endif 
	
	lits.push((stats_sequance[a_tmp]==l_True)?~Lit( b_tmp ):Lit( b_tmp ));
      }else{
	//printf("-");
      }
      a_tmp++;
      b_tmp++;
    }
    S.addClause(lits);
#ifdef PRINT_STATE
    //puts(" ");
#endif  
}

void add_clauses_with_variable_shift( std::vector< std::vector<Lit> > &clauses2add, Solver &S,int shift){

  unsigned i;
  vec<Lit> lits;
  lits.clear();  

  i=clauses2add.size();
  while(i--){
    int j=clauses2add[i].size();
    while(j--){
      lits.push(toLit( toInt(clauses2add[i][j]) + (shift+shift) ));
    }

    S.addClause(lits);
    lits.clear();    
  }
 
}

void construct_depth(unsigned k){

  int i;
  int depth_left_to_build = k - global_var.depth;
  Solver *S=global_var.S;

  if( depth_left_to_build < 0 ) return;
   
  i = global_var.number_of_var * depth_left_to_build;
  while(i--){
    (*S).newVar();
  }

  for(i=global_var.depth-1; i< (k - 1); i++){
    add_clauses_with_variable_shift(*global_var.orig_clauses, *S, i*global_var.number_of_var);
  }

  global_var.depth=k;  
}



/*Contain help text dispayed if arguments are incorrect or help is envoked*/
void print_help(){

  puts(" Usage: bns [-l N|-u N] file ");
  puts("");
  puts(" -l N     restricts the search for atractors of lenght N and its factors.   "); 
  puts("          For example \"bns -l 6\" searches only for atractors of           ");
  puts("          length 6 and its factors 3,2, and 1. Atractors of length 4,5 and  ");
  puts("          any other of length larger than 6 will not be found.              ");
  puts("          Limiting the search space might drastically reduce runtime and can");
  puts("          be an option to try when unrestricted search does not finish.     ");
  puts(" -u N     set initial unfolding of transition relation to N levels instead  "); 
  puts("          of the default value which is equal to number of variables in     ");
  puts("          Boolean network. This option does not impact the result but might ");
  puts("          increase or reduce total runtime. It is should not be used        ");
  puts("          together with -l option.                                          ");
  puts(" file     the file name describing Boolean network for which atractors are  ");
  puts("          searched for.");
} 

int 
main(int argc, char *argv[]) 
{

  int i,j,n;
  Solver      S;
  unsigned atractor_count=0;
  unsigned atractor_stats_count=0;
  vec<Lit> lits;
  std::vector<Lit> lits_std;
  char command[100];
  float avg_length = 0;

  std::vector< std::vector<Lit> > orig_clauses;
  global_var.orig_clauses = &orig_clauses;


  if(argc==1 || argc==3 || argc> 4 )
    {
      print_help();
      exit(0);
    }

  if(argc==2)
    ReadNET(argv[1], S);

  i=1;
  while(i<argc){
    if(strcmp(argv[i], "-l")==0){
      i++;
      if(i==argc)
       {
        print_help();
        exit(0);
        }
      exact_atractor_size = atoi(argv[i]);
     if(exact_atractor_size<1)
       {
        printf("Error: Wrong -l paramiter equal to %d. The value must be above 0.\n\n",exact_atractor_size);
        print_help();
        exit(0);
        }
        i++;
        continue;
    }

    if(strcmp(argv[i], "-u")==0){
      i++;
      if(i==argc)
       {
        print_help();
        exit(0);
        }
      init_unfolding = atoi(argv[i]);
     if(init_unfolding<2)
       {
        printf("Error: Wrong -u paramiter is equal to %d. The value must be above 1.\n\n",init_unfolding);
        print_help();
        exit(0);
        }
        i++;
        continue;
    }

    ReadNET(argv[i], S); 
    break;
  }

  if(i==argc)
    {
    print_help();
    exit(0);
    }

    
  lits_std.clear();


  /*Handle assignments of variables made in solver*/
  i=number_of_var;
  while(i--){
    if(S.value(i)!= l_Undef){
      lits_std.push_back((S.value(i)==l_True)? Lit(i) : ~Lit(i));
      orig_clauses.push_back( lits_std );
      lits_std.clear();
    }
  }
 

  global_var.S = &S;
  global_var.number_of_var = number_of_var;
  global_var.depth=2;

  if(exact_atractor_size==0){
    if(init_unfolding==0){
      if(number_of_var<100){
        construct_depth(number_of_var);
      }else{
        construct_depth(100);
      }
    }else{
      construct_depth(init_unfolding);
    }
  }else{
    construct_depth(exact_atractor_size+1);

    /*Adding requirement state s_0 is equal to state s_exact_atractor_size*/
    i=number_of_var;
    lits.clear();

    while(i--){     
      lits.push( Lit( i ) );
      lits.push( ~Lit( i + exact_atractor_size*number_of_var ) );
      S.addClause(lits);
      lits.clear();
      lits.push( ~Lit( i ) );
      lits.push( Lit( i + exact_atractor_size*number_of_var ) );
      S.addClause(lits);
      lits.clear();
    }
  }

  if(exact_atractor_size==0)
    puts("Start searching for all atractors.");
  else
    printf("Start searching for atractors of length %d or its factors.\n", exact_atractor_size);

  while(1){

    if (!S.solve()) break;
    

    for( i=1; i<global_var.depth; i++ ){
	
      if(compare_states(0,i,number_of_var,S.model )){
	atractor_count++;
	atractor_stats_count+=i;
	//PRINT(atractor_stats_count);
	//constrain all states of atractor sequence on 0 index variable
	for( j = i-1; j >= 0; j-- ){
	  constrain_state(j, S.model, 0, S, number_of_var );
#ifdef PRINT_STATE
  	int a_tmp=j*number_of_var;	
	  int counter=number_of_var;
	  while(counter--){
	      if (S.model[a_tmp] != l_Undef){
		printf( "%s", (S.model[a_tmp]==l_True)?"1":"0");
	      }else{
		printf("-");
	      }
	      a_tmp++;
	  }
        printf("\n");
#endif 
	}

	printf("Attractor number:%d The size of atractor:%d\n",atractor_count,i);
	avg_length = avg_length + i;
	break;
      }
    }

    if( global_var.depth == i ){ // if exact_atractor_size==0 we never supose to reach here since atractor either found or unSat
      construct_depth( (global_var.depth*3 >> 1)+1 );
      //printf("Depth of unfolding transition relation is increased to %d\n",global_var.depth);
    }
  }

  printf("%d attractors of ",atractor_count);
  printf("average length %0.2f\n",avg_length/(float)atractor_count);


  //printf("%d %d %0.2f ",n,atractor_count,avg_length/(float)atractor_count);

  //PRINT(atractor_count);
  //PRINT(atractor_stats_count);
  //PRINT(global_var.depth);
  //printf(ret ? "SATISFIABLE\n" : "UNSATISFIABLE\n");
  return 0;
  
}

inline void add2orig_clause(vec<Lit> &lits,std::vector<Lit> &lits_std){


           for(int j=0; j<lits.size();j++){
           	//PRINT(toInt(lits[j]));
           	lits_std.push_back(lits[j]);
        	}
           (*global_var.orig_clauses).push_back( lits_std );
           lits_std.clear();
}

/* Read in NET file and returns 2^n transition function*/
void
ReadNET(char *fileName, Solver& S)
{

  char temp[MAX_LENGTH];
  char slask[MAX_LENGTH];
  int i , k , tmp_1;

  int input_array[MAX_NUMBER_OF_INPUTS];
  int number_of_inputs,current_number_of_inputs=0;
  int current_var;
  FILE *fp,*fp_temp;      
  vec<Lit> lits;
  std::vector<Lit> lits_std;


  unsigned line_count=0;

//the following allows counting of line
#define fgets(X,Y,Z);  fgets((X),(Y),(Z));line_count++;  

  /* Open NET file for reading */
  if ((fp = fopen(fileName, "r")) == NULL) {
    fprintf(stderr, "Couldn't open NET file: %s\n", fileName);
    exit (1);
  }

  /* Reading number of non-redundent var*/
  while(!feof(fp)) {
    fgets(temp,MAX_LENGTH, fp);
    if (strncmp(temp,".v",2)==0) {
      /* Find size of the cubes */
      sscanf(temp, "%s %i", slask, &number_of_var);
      break;
    }
  }
  var2red_var=(int*) calloc(number_of_var+1, sizeof(int));

  size_nonred_array=number_of_var;



  redunt_number_of_var=number_of_var;
  redunt_var2var=(int*) calloc(redunt_number_of_var+1, sizeof(int));


  number_print_var=number_of_var;

  int count_nodes;
  for(count_nodes=1; count_nodes<=number_of_var; count_nodes++ ){
    redunt_var2var[count_nodes]=count_nodes;
    var2red_var[count_nodes]=count_nodes;
  }

 
  if(feof(fp)) {
    fprintf(stderr, "Wrong format of input file. End of file is reached but no node function is red" );
    exit(1);
  }
  
  k=number_of_var;
  while(k--){
    S.newVar();S.newVar();
  }
  
  //RBN_init_global_var(bdd_manager);

  /*-------------------- Filling information about nodes  -------------------*/


  while(!feof(fp)){
    fgets(temp,MAX_LENGTH, fp);
  next_vertex:
    if (strncmp(temp,".n",2)==0) {
      /* Find size of the cubes */

      i=3;
      sscanf(&temp[i],"%d",&current_var);
      

      if( current_var < 0 || current_var >redunt_number_of_var){
	fprintf(stderr, "Wrong format of input file in line %d.The varible %d in string:\n %s exceeds number of declared variables.\n",line_count, current_var,temp);
	exit (1);
      }
      

      i++;
      /* go to first space */
      while(temp[i]!=' '){
	if(temp[i]=='\n'){
	  fprintf(stderr, "Wrong format of input file in line %d. Wrong format of the string: %s\n",line_count, temp );
	  exit (1);
	}
	i++;
      }
      i++;
      /* go to the end of sequense of spaces */
      while(temp[i]==' '){
	i++;
      }
      if(temp[i]=='\n'){
	fprintf(stderr, "Wrong format of input file in line %d. Wrong format of the string: %s\n",line_count, temp );
	exit (1);
      }
      sscanf(&temp[i],"%d",&number_of_inputs);

      if(number_of_inputs > MAX_NUMBER_OF_INPUTS){
	fprintf(stderr, "Wrong format of input file in line %d . Number of inputs exceeds allowed number of inputs %s\n",line_count, temp  );
	exit (1);
      }
      

      while(1){
	i++;
	while(temp[i]!=' '){
	  if(temp[i]=='\n'){
	    goto end_loops;
	  }
	  i++;
	}
	i++;
	/* go to the end of sequense of spaces */
	while(temp[i]==' '){
	  i++;
	}
	if(temp[i]=='\n'){
	  goto end_loops;
	}
	sscanf(&temp[i],"%d",&tmp_1);

	if( tmp_1 < 0 || tmp_1 >redunt_number_of_var){
	  fprintf(stderr,"Wrong format of input file in line %d. The varible %d in string:\n %s exceeds number of declared variables.\n",line_count, tmp_1,temp);
	  exit (1);
	}
	
	if( redunt_var2var[tmp_1] == 0){
	  fprintf(stderr, "Wrong format of input file in line %d. One of the input was not declared in list of inputs %s\n",line_count, temp  );
	  exit (1);
	}


	input_array[current_number_of_inputs++]= redunt_var2var[tmp_1];


      }
    end_loops:


      if(current_number_of_inputs!=number_of_inputs){
	fprintf(stderr, "Wrong format of input file in line %d. Declared number of inputs is not equal to actual number of input %s\n",line_count, temp  );
	exit (1);
      }

      if(number_of_inputs==0){
	  fgets(temp,MAX_LENGTH, fp);
	  
	  if( temp[0]=='1'){
	    lits.push( Lit( current_var - INDEX_OF_FIRST_VARIABLE ) );
	  }else{
	    if( temp[0]!='0'){
	      printf("Node %d assumed to be constant 0\n",current_var);
	    }
	    lits.push( ~Lit( current_var - INDEX_OF_FIRST_VARIABLE ) );
	  }
	  S.addClause(lits);
	  add2orig_clause(lits,lits_std);
	  lits.clear();
	  goto next_vertex;
      }


      /*---------------  Reading in the function of the node ---------------*/
      

      current_number_of_inputs=0;

      while(!feof(fp)){
	fgets(temp,MAX_LENGTH, fp);
	if( temp[0]=='0' || temp[0]=='1' || temp[0]=='-'){

	  for(i=0;i<number_of_inputs;i++){
	    if(temp[i]=='-')
	      continue;

	    lits.push( (temp[i]=='1') ? ~Lit( number_of_var + input_array[i] - INDEX_OF_FIRST_VARIABLE) : Lit(number_of_var + input_array[i] - INDEX_OF_FIRST_VARIABLE) );

	    
	    if(temp[i]!='1' && temp[i]!='0'){
	      fprintf(stderr, "Wrong format of input file in line %d. Wrong line:%s\n",line_count, temp  );
	      exit (1);	
	    }      
	  }

	  ASSERT((number_of_var + current_var - INDEX_OF_FIRST_VARIABLE) < S.nVars());


	  i++;
	  if(temp[i]=='1'){
	    lits.push( Lit(  current_var - INDEX_OF_FIRST_VARIABLE ) );
	  }else{
	    if(temp[i]=='0'){
	      lits.push( ~Lit( current_var - INDEX_OF_FIRST_VARIABLE ) );
	    }else{
	      fprintf(stderr, "Wrong format of input file in line %d. Unexpected charecter in output of minterm in line:%s\n",line_count, temp  );
	      exit (1);	
	    }
	  }
	  S.addClause(lits);
	
	  add2orig_clause(lits,lits_std);

	  lits.clear();	 

	}else{

	  if( redunt_var2var[current_var] == 0){
	    fprintf(stderr, "Wrong format of input file.The node:%d was not declared \n", current_var  );
	    exit (1);
	  }
	  
	  break;
	  
	}	
      }
	  
      if(!feof(fp)){
	goto next_vertex;
      }

      /*We are here after we reach the end of the file*/
      if( redunt_var2var[current_var] == 0){
	fprintf(stderr, "Wrong format of input file.The node:%d was not declared \n", current_var  );
	exit (1);
      }

      
    }/* -------- end processing input informaition of a vertex ------------*/
  }
/*end processing the file*/
  
  fclose(fp);
  
#undef fgets

}
