package enforcer.rt;

/* $Id: Reflect.java 580 2007-05-30 03:07:12Z cartho $ */

import enforcer.etc.Options;
import enforcer.instr.CoverageTransformer;
import enforcer.log.Log;
import java.lang.reflect.Array;
import java.lang.reflect.Method;

/** Helper class: Advanced usage of reflection. */

public class Reflect {
    static boolean suppressArtificialException =
        Options.getBool("supprArtificialExc");

    /** Prints stack trace if it is not equal to the "magic string". */
    public static void printStackTrace(Exception exc) {
        String msg = exc.getMessage();
        if (suppressArtificialException && ((msg == null) ||
                                            (msg.equals(CoverageTransformer.ENFORCER_EXC_MSG)))) {
            return;
        }
        exc.printStackTrace();
    }

    /** @return class descriptor or <tt>null</tt> if class not found. */
    static Class clsByName(String name) {
        name = name.replace('/', '.');
        try {
            return Class.forName(name);
        } catch (ClassNotFoundException e) {
            Log.log("Could not find class desc for " + name, Log.ERROR);
            return null;
        }
    }

    /** Get class descriptor of class referred to by "name".
      * Also works on primitive types. 
      * @return class descriptor or <tt>null</tt> for failed lookup. */
    public static Class getClsDesc(String name) {
        int strlen = name.length();
        if (strlen == 1) {
            switch(name.charAt(0)) {
            case 'B':
                return byte.class;
            case 'C':
                return char.class;
            case 'D':
                return double.class;
            case 'F':
                return float.class;
            case 'I':
                return int.class;
            case 'J':
                return long.class;
            case 'S':
                return short.class;
            default:
                assert (name.charAt(0) == 'Z');
                return boolean.class;
            }
        }
        if ((name.charAt(0) == 'L') && (name.charAt(strlen - 1) == ';')) {
            return clsByName(name.substring(1, strlen - 1));
        }
        // primitive types
        if (name.equals("byte"))
            return byte.class;
        if (name.equals("char"))
            return char.class;
        if (name.equals("double"))
            return double.class;
        if (name.equals("float"))
            return float.class;
        if (name.equals("int"))
            return int.class;
        if (name.equals("long"))
            return long.class;
        if (name.equals("short"))
            return short.class;
        if (name.equals("boolean"))
            return boolean.class;
        if (name.charAt(0) != '[') { // normal class name
            return clsByName(name);
        }
        // final case: array
        assert (name.charAt(0) == '[');
        int ndim = name.lastIndexOf('[') + 1;
        int[] dims = new int[ndim];
        Class arrayType = getClsDesc(name.substring(ndim));
        if (arrayType == null) {
            return null;
        }
        Object arr = Array.newInstance(arrayType, dims);
        return arr.getClass();
    }

    public static boolean checkMethodExc(Method method, String excName,
                                         int excBlock) {
        Class[] exceptions = method.getExceptionTypes();
        Class expectedExc = getClsDesc(excName);
        for (int i = 0; i < exceptions.length; i++) {
            if (expectedExc.isAssignableFrom(exceptions[i])) {
                return false;
            }
        }
        Coverage.unChecked.set(excBlock);
        if (Coverage.numBlocks <= excBlock) {
            Coverage.numBlocks = excBlock + 1;
        }
        Log.log("Method " + method.getDeclaringClass().getName() +
                "." + method.getName() +
                " does not actually throw " + excName, Log.DEBUG);
        return true;
    }
}
