package enforcer.instr;

/* $Id: CoverageTransformer.java 580 2007-05-30 03:07:12Z cartho $ */

import enforcer.etc.Options;
import enforcer.log.Log;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintWriter;
import java.lang.reflect.Constructor;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import serp.bytecode.BCClass;
import serp.bytecode.BCField;
import serp.bytecode.BCMethod;
import serp.bytecode.ClassInstruction;
import serp.bytecode.Code;
import serp.bytecode.ConstantInstruction;
import serp.bytecode.Exceptions;
import serp.bytecode.ExceptionHandler;
import serp.bytecode.IfInstruction;
import serp.bytecode.Instruction;
import serp.bytecode.JumpInstruction;
import serp.bytecode.LocalVariable;
import serp.bytecode.LocalVariableTable;
import serp.bytecode.MethodInstruction;
import serp.bytecode.Project;

public class CoverageTransformer {
    public final static String ENFORCER_EXC_MSG =
        "Artificial exception generated for coverage";
    final static String NEW_TEST_SUPER_CLASS =
        "enforcer.rt.ExceptionTestCase";

    static class InstrSetEntry implements Comparable<InstrSetEntry> {
        Instruction ins;	/* instruction where method call occurs */
        Class excType;		/* tupe of exception to be instrumented for */
        Constructor excCons;	/* constructor of said exception */
        int rank;		/* rank (row) in exception table */

        public int compareTo(InstrSetEntry other) {
            int res = this.ins.getByteIndex() - other.ins.getByteIndex();
            if (res != 0) {
                return res;
            }
            res = this.rank - other.rank;
            if (res != 0) {
                return res;
            }
            return System.identityHashCode(excType) -
                   System.identityHashCode(other.excType);
        }

        public boolean equals(Object obj) {
            if (! (obj instanceof InstrSetEntry)) {
                return false;
            }
            return (this.compareTo((InstrSetEntry) obj) == 0);
        }
    }

    public static int excBlock = 0;
    private LocalVariable excBlockLocalVar;
    private LocalVariable invocationTarget;
    private LocalVariable dynCheckResult;
    BCMethod currentMethod;
    TreeSet<InstrSetEntry> instrSet;
    HashSet<Instruction> handlerSet;
    boolean dynExcCheck = Options.getBool("dynExcCheck");

    Constructor getExcCons(Class cls) {
        Class[] args = { String.class };
        try {
            return cls.getConstructor(args);
        } catch (NoSuchMethodException e) {
            // second try is below: using second arg of type Throwable
        }
        Class[] args2 = { String.class, Throwable.class };
        try {
            return cls.getConstructor(args2);
        } catch (NoSuchMethodException e) {
            // next try is below: using second arg of type Throwable
        }
        Class[] args3 = { Throwable.class, String.class };
        try {
            return cls.getConstructor(args3);
        } catch (NoSuchMethodException e) {}
        return null;
    }

    private void addArg(boolean str, Code code, String[] args, int pos) {
        ConstantInstruction excArg = code.constant();
        if (str) {
            args[pos] = "Ljava/lang/String;";
            excArg.setValue(ENFORCER_EXC_MSG);
        } else {
            code.constant().setNull();
            args[pos] = "Ljava/lang/Throwable;";
        }
    }

    boolean addExcCheck(Instruction curr) {
        if ((curr.getOpcode() == 0xb8) || // 0xb8 = invokestatic
                (curr.getOpcode() == 0xb7)) { // 0xb7 = invokespecial
            return false;
        }
        return dynExcCheck;
    }

    Instruction findInstanceCalled(Instruction curr, Code code) {
        Instruction first;
        // find instance called by current method instruction
        // (invoke...) and insert instructions copying that instance
        // reference to a local variable.
        MethodInstruction invocation = (MethodInstruction) curr;
        int stackHeight = invocation.getMethodParamTypes().length;
        // loop back to prev. ins until stack heights match
        while (stackHeight != 0) {
            curr = code.previous();
            stackHeight -= curr.getStackChange();
        }
        code.before(curr);
        first = code.dup();
        if (invocationTarget == null) {
            invocationTarget = addLocalVar(code, "$invocationTarget",
                                           "Ljava/lang/Object;");
        }
        code.astore().setLocalVariable(invocationTarget);
        // store ref., retrieve it later
        return first;
    }

    Instruction addDynamicExcTypeCheck(Code code, Instruction curr,
                                       Class excType) {
        Instruction target;
        BCMethod method = ((MethodInstruction) curr).getMethod();
        if (dynCheckResult == null) {
            dynCheckResult = addLocalVar(code, "$dynCheckResult", "Z");
        }
        if (method == null) {
            BCMethod codeMeth = (BCMethod) code.getOwner();
            Log.log(codeMeth.getDeclarer().getName() + "." +
                    codeMeth.getName() +
                    ": Cannot obtain method descriptor for " +
                    ((MethodInstruction) curr).getMethodName(),
                    Log.WARNING);
            target = code.constant().setValue(0);
            code.istore().setLocalVariable(dynCheckResult);
            return target;
        }
        // Retrieve reference to method being called
        // (1) get reference to instance being invoked
        target = code.aload().setLocalVariable(invocationTarget);
        // retrieve reference to object's class descriptor
        code.invokevirtual().setMethod("java/lang/Object", "getClass",
                                       "Ljava/lang/Class;", new String[0]);
        // (2) method name
        code.constant().setValue(method.getName());
        // (3) obtain argument list (as strings) of static type
        String[] methodParams = method.getParamNames();
        // (3.a) create new array object to hold class descriptors
        code.constant().setValue(methodParams.length);
        code.anewarray().setType("Ljava/lang/Class;");
        // (3.b) For each arg element, create class descriptor
        String[] reflParams = new String[] { "Ljava/lang/String;" };
        for (int i = 0; i < methodParams.length; i++) {
            code.dup();
            code.constant().setValue(i);
            // obtain class descriptor via helper method
            // string constant describing the date type
            code.constant().setValue(methodParams[i]);
            // obtain class descriptor from string
            code.invokestatic().setMethod("enforcer/rt/Reflect",
                                          "getClsDesc",
                                          "Ljava/lang/Class;",
                                          reflParams);
            code.aastore();
        }
        // (4) use class descriptor, method name and args to get method desc.
        String[] reflectParams = new String[2];
        reflectParams[0] = "Ljava/lang/String;";
        reflectParams[1] = "[Ljava/lang/Class;";
        MethodInstruction methLookup = code.invokevirtual();
        methLookup.setMethod("java/lang/Class", "getDeclaredMethod",
                             "Ljava/lang/reflect/Method;",
                             reflectParams);
        // exception handler for call to getMethod
        JumpInstruction toEndOfHandler = code.go2();
        Instruction handler = code.dup();
        code.invokevirtual().setMethod("java/lang/NoSuchMethodException",
                                       "printStackTrace", "V", new String[0]);
        code.athrow();
        // end of exception handler code

        code.addExceptionHandler(methLookup, toEndOfHandler, handler,
                                 "java/lang/NoSuchMethodException");
        // second argument: name of exception
        toEndOfHandler.setTarget(code.constant().setValue(excType.getName()));

        // final argument: exception block number
        code.constant().setValue(excBlock);
        String[] p = new String[] { "Ljava/lang/reflect/Method;",
                                    "Ljava/lang/String;", "I" };
        // Compare run-time type of exceptions thrown by method
        code.invokestatic().setMethod("enforcer/rt/Reflect",
                                      "checkMethodExc", "Z", p);
        code.istore().setLocalVariable(dynCheckResult);
        return target;
    }

    // Perform run-time exception signature check.
    // Insert code to check whether the method in question really
    // throws the exception that was detected/expected at compile-time.
    void addExcLauncher(Code code, InstrSetEntry currItem, boolean chain) {
        Instruction curr = currItem.ins;
        Class excType = currItem.excType;
        Constructor excCons = currItem.excCons;
        Instruction first = null;
        String className = code.getMethod().getDeclarer().getName();
        code.before(curr);
        // Find out run-time type of method being called. This requires
        // finding out the first argument (the instance being called).
        // Instructions have to be traced back until the one
        // instruction is found whose result corresponds to
        // the first argument.
        if (addExcCheck(curr)) {
            first = findInstanceCalled(curr, code);
        }

        // record current excBlock in local variable (for "generic" handler)
        if (first != null) {
            first = code.constant().setValue(excBlock);
        } else {
            code.constant().setValue(excBlock);
        }
        code.istore().setLocalVariable(excBlockLocalVar);
        // test whether we re-run code or record coverage
        code.getstatic().setField("enforcer.rt.Eval", "reRun", "I");
        code.constant().setValue(excBlock);
        IfInstruction branchExc = code.ificmpeq();
        code.getstatic().setField("enforcer.rt.Eval", "excPath", "I");
        code.constant().setValue(excBlock);
        IfInstruction branchReRun = code.ificmpne();

        // generate artificial exception
        ClassInstruction excGen = code.anew();
        excGen.setType(excType.getName());
        code.dup();

        Class[] params = excCons.getParameterTypes();
        int len = params.length;
        String[] consArgs = new String[len];
        addArg((params[0] == String.class), code, consArgs, 0);
        if (len == 2) {
            // add second argument (Throwable null)
            addArg((params[1] == String.class), code, consArgs, 1);
        }

        // actual <init> call
        MethodInstruction excInit = code.invokespecial();
        excInit.setMethod(excType.getName(), "<init>", "V", consArgs);
        code.athrow(); // throw artificial exc.
        Instruction target = curr;

        if (addExcCheck(curr)) {
            target = addDynamicExcTypeCheck(code, curr, excType);
        }
        // set jump targets
        branchExc.setTarget(excGen);
        if (chain) {
            // special case: several handlers for same instruction
            // make sure jump points to the first instruction of
            // the next instrumented block
            branchReRun.setTarget(code.nop());
        } else {
            branchReRun.setTarget(target);
        }
        //fixGoto(code, first);
    }

    void recordTest(Instruction curr) {
        IfInstruction branch = null;
        Code code = curr.getCode();
        // record test
        code.after(curr);
        // If exc. sig. is to be run-time checked, use information
        // from flag (which was instrumented) and skip constant value ins.
        // and Coverage.recordTest call.
        if (addExcCheck(curr)) {
            code.iload().setLocalVariable(dynCheckResult);
            branch = code.ifne();
        }
        ConstantInstruction coverage = code.constant();
        coverage.setValue(excBlock);
        MethodInstruction covRecord = code.invokestatic();
        String[] args = new String[1];
        args[0] = "I";
        covRecord.setMethod("enforcer.rt.Coverage", "recordTest", "V", args);
        if (addExcCheck(curr)) {
            branch.setTarget(code.next());
        }
    }

    void fixGoto(Code code, Instruction first) {
        // test whether a goto target affects the successor instruction
        // if so, it should instead go to the first instruction of
        // our instrumented code.
        Instruction target = code.next();
        Instruction[] insns = code.getInstructions();
        for (int i = 0; i < insns.length; i++) {
            if ((insns[i] instanceof JumpInstruction) &&
                    (((JumpInstruction)insns[i]).getTarget() == target)) {
                Log.log("Adjusting target for instr. " +
                        insns[i].getByteIndex() + " from " +
                        target.getByteIndex() + " to " +
                        first.getByteIndex());
                ((JumpInstruction)insns[i]).setTarget(first);
            }
        }
    }

    void instrumentHandler(Code code, Instruction handler) {
        code.after(handler);
        code.getstatic().setField("enforcer.rt.Coverage", "exc",
                                  "Ljava/util/BitSet;");
        code.iload().setLocalVariable(excBlockLocalVar);
        String[] args = new String[1];
        args[0] = "I";
        MethodInstruction covRecord = code.invokevirtual();
        covRecord.setMethod("java.util.BitSet", "set", "V", args);
        /* debug output
        	code.getstatic().setField("java.lang.System", "out",
        				  "Ljava/io/PrintStream;");
        	code.iload().setLocalVariable(excBlockLocalVar);
        	code.invokevirtual().setMethod("java.io.PrintStream", "println",
        				       "V", args);
        */
    }

    void instrumentHandlers(Code code) {
        Iterator<Instruction> it = handlerSet.iterator();
        while (it.hasNext()) {
            instrumentHandler(code, it.next());
        }
    }

    boolean checkInvocationArg(MethodInstruction ins) {
        if (ins.getMethodName().compareTo("run") != 0) {
            return false;
        }
        String className = ins.getMethodDeclarerName();
        if (!((className.startsWith("junit.")) &&
                (className.endsWith(".TestRunner")))) {
            return false;
        }
        // allow for two ways to run tests:
        // 1) static public TestResult run(Test)
        // 2) static public void run(Class)

        String returnType = ins.getMethodReturnName();
        String[] args = ins.getMethodParamNames();
        String[] newArgs = new String[1];
        if ((returnType.compareTo("junit.framework.TestResult") == 0) &&
                (args.length == 1) &&
                (args[0].compareTo("junit.framework.Test") == 0)) {
            // replace invocation target with wrapped runner
            ins.setMethod("enforcer.rt.WrappedRunner", "run",
                          returnType, args);
            return true;
        }
        if ((returnType.compareTo("void") == 0) &&
                (args.length == 1) &&
                (args[0].compareTo("java.lang.Class") == 0)) {
            // replace invocation target with wrapped runner
            ins.setMethod("enforcer.rt.WrappedRunner", "run",
                          returnType, args);
            return true;
        }
        return false;
    }

    boolean transformTestLaunch(Code code) {
        Instruction[] insns = code.getInstructions();
        boolean modified = false;
        for (int i = 0; i < insns.length; i++) {
            if (insns[i].getOpcode() == 0xb8) { // invokestatic
                MethodInstruction ins = (MethodInstruction) insns[i];
                String className = ins.getMethodDeclarerName();
                if (!checkInvocationArg(ins)) {
                    continue;
                }
                Log.log("Found test launch insn!");

                code.before(ins);
                // add initialization code
                code.constant().setValue(className);
                MethodInstruction invoke = code.invokestatic();
                String[] launchArgs = new String[1];
                launchArgs[0] = "Ljava/lang/String;";
                invoke.setMethod("enforcer.rt.Eval", "setTestRunner",
                                 "V", launchArgs);

                invoke = code.invokestatic();
                String[] noArgs = new String[0];
                invoke.setMethod("enforcer.rt.Coverage", "initialize",
                                 "V", noArgs);
                code.after(ins);

                // add reRun invocation
                invoke = code.invokestatic();
                invoke.setMethod("enforcer.rt.Eval", "reRunTests",
                                 "Z", noArgs);
                // add while loop
                code.ifeq().setTarget(invoke);

                code.calculateMaxStack();
                modified = true;
                break;
            }
        }
        return modified;
    }

    boolean checkSuppressionList(BCMethod method) {
        if (method.isSynthetic()) {
            return true;
        }
        if (method.getName().compareTo("<clinit>") == 0) {
            return true;
        }
        // a) clinit should not fail (causes VM abortion)
        // b) cannot be executed twice
        return false;
    }

    @SuppressWarnings("unchecked")
    Class<Class> uncheckedCastToClass_Class(Class c) {
        return (Class<Class>)c;
    }

    /** Returns <tt>true</tt> if exception <tt>exc</tt> can be
        handled by handler of type <tt>handlerType</tt>. */
    static boolean excIsHandledBy(Class exc, Class handlerType) {
        return handlerType.isAssignableFrom(exc);
    }

    /** Returns <tt>true</tt> if instruction is already <em>completely</em>
        protected by a handler "higher up" (with lower index) in the
    exception table. Complete protection means all exceptions
    thrown are handled by higher (nested) handlers. Also returns
    <tt>true</tt> if handler is not needed.
    @param ins: instruction (for getting PC).
    @param excTypes: array of types of exceptions thrown by
    <tt>ins</tt>.
    @param handler: array of handlers in this method.
    @param idx: index of current handler; all handlers higher
    up are checked.
    @return <tt>true</tt> if handler is not used.
     */
    boolean insAlreadyProtected(Instruction ins, Class[] excTypes,
                                ExceptionHandler[] handler, int idx) {
        int pc = ins.getByteIndex();
        /* check for any uncaught exceptions further up in chain */
        for (int i = 0; i < excTypes.length; i++) {
            Class excType = excTypes[i];
            boolean isProtected = false;
            for (int j = 0; j < idx; j++) {
                ExceptionHandler hdlr = handler[j];
                Class<Class> catchType =
                    uncheckedCastToClass_Class(hdlr.getCatchType());
                if (catchType == null) {
                    /* handler for finally block, ignore */
                    continue;
                }
                if (excIsHandledBy(excType, catchType) &&
                        (hdlr.getTryStartPc() <= pc) &&
                        (hdlr.getTryEndPc() > pc)) {
                    isProtected = true;
                }
            }
            if (!isProtected &&
                    excIsHandledBy(excType, handler[idx].getCatchType())) {
                Log.log("Genuine handler for " + excType.getName() +
                        "/" + handler[idx].getCatchType().getName(),
                        Log.DEBUG_LOWLEVEL);
                return false;
            }
        }
        return true;
    }

    static BCMethod getSuperClassMethod(BCClass cls, String methodName,
                                        Class[] params) {
        BCMethod method = null;
        BCClass superClass;
        superClass = cls.getSuperclassBC();
        while ((method == null) && (superClass != null)) {
            method = superClass.getDeclaredMethod(methodName, params);
            superClass = superClass.getSuperclassBC();
        }
        if (method == null) {
            BCClass[] interfaces = cls.getInterfaceBCs();
            int i = 0;
            while ((method == null) && (i < interfaces.length)) {
                method = interfaces[i].getDeclaredMethod(methodName, params);
                if (method == null) {
                    method = getSuperClassMethod(interfaces[i],
                                                 methodName, params);
                }
                i++;
            }
        }
        return method;
    }

    /** Returns true if an exception of the type registered in
        <tt>handler[idx]</tt> or its subtype can be thrown by instruction
        <tt>ins</tt> and caught by <tt>handler[idx]</tt>.
    @param <tt>ins</tt>: Instruction to be analyzed.
    @param <tt>handler[]</tt>: Array of handlers in this method.
    @param <tt>idx</tt>: Index of current handler.
    @param <tt>excCons</tt>: Constructor for exception (if
    instrumentation necessary at later stage). */
    boolean checkExceptionsInIns(Instruction ins,
                                 ExceptionHandler[] handler, int idx,
                                 Constructor excCons) {
        // check type of instruction to "simulate" exceptions if necessary
        // FieldInstruction: Possible NullPointerException.
        // Does not require an exception handler, can be tested in other ways
        // => no instrumentation
        // same for anything but method calls (for now)
        if (!(ins instanceof MethodInstruction)) {
            return false;
        }

        Class<Class> excType =
            uncheckedCastToClass_Class(handler[idx].getCatchType());
        // interesting case: check exceptions thrown by method
        MethodInstruction mIns = (MethodInstruction) ins;
        BCMethod meth = (mIns).getMethod();
        // Special case:
        // Method signature not available: assume exception of type
        // that is handled by catch clause
        if (meth == null) {
            meth = getSuperClassMethod(mIns.getMethodDeclarerBC(),
                                       mIns.getMethodName(),
                                       mIns.getMethodParamTypes());
        }
        if (meth == null) {
            Log.log(ins.getByteIndex() + ": " + ins.getName() + " " +
                    ins.getCode().getMethod().getDeclarer().getName() +
                    "." + ins.getCode().getMethod().getName() +
                    ": Could not resolve exceptions for method " +
                    ((MethodInstruction) ins).getMethodDeclarerName() +
                    "." + ((MethodInstruction) ins).getMethodName() +
                    "; assuming " + excType.getName() + ".", Log.INFO);
            addToInstrSet(ins, excType, excCons, idx);
            handlerSet.add(handler[idx].getHandlerStart());
            return true;
        }

        // no exceptions thrown by called method
        Exceptions exceptions = meth.getExceptions(false);
        if (exceptions == null) {
            return false;
        }

        Class[] exc = exceptions.getExceptionTypes();

        if (insAlreadyProtected(ins, exc, handler, idx)) {
            return false;
        }

        for (int i = 0; i < exc.length; i++) {
            if (excIsHandledBy(exc[i], excType)) {
                Log.log(ins.getByteIndex() + ": " + ins.getName() + " " +
                        ((MethodInstruction) ins).getMethodDeclarerName() +
                        "." + ((MethodInstruction) ins).getMethodName() +
                        " throws " + exc[i].getName(),
                        Log.DEBUG_LOWLEVEL);
                addToInstrSet(ins, exc[i], excCons, idx);
                handlerSet.add(handler[idx].getHandlerStart());
                return true;
            }
        }
        // no matching exceptions thrown by called method
        return false;
    }

    void instrumentThrowableIns(Code code) {
        InstrSetEntry curr, next;
        boolean eq;
        Iterator<InstrSetEntry> it = instrSet.iterator();
        if (!it.hasNext()) {
            return;
        }
        next = it.next();
        while (it.hasNext()) {
            curr = next;
            next = it.next();
            eq = next.ins.getByteIndex() == curr.ins.getByteIndex();
            addExcLauncher(code, curr, eq);
            recordTest(curr.ins);
            excBlock++;
        }
        // next = final ins; no successor with equal byte code idx
        addExcLauncher(code, next, false);
        recordTest(next.ins);
        excBlock++;
    }

    boolean checkExceptions(Code code, ExceptionHandler[] handler, int idx) {
        boolean modified = false;
        ExceptionHandler currHandler = handler[idx];
        Class excType = currHandler.getCatchType();
        if (excType == null) {
            return false;
        }
        // check for constructor before entering loop
        Constructor excCons = getExcCons(excType);
        if (excCons == null)  {
            Log.log(currentMethod.getDeclarer().getName() + "." +
                    currentMethod.getName() +
                    ": Skipping exception of type", Log.WARNING);
            Log.log(currHandler.getCatchName() +
                    ": Cannot find proper constructor.", Log.WARNING);
            return false;
        }
        Instruction end = currHandler.getTryEnd();
        code.before(currHandler.getTryStart());
        Instruction curr = code.next();
        while (curr != end) {
            modified = checkExceptionsInIns(curr, handler, idx, excCons)
                       || modified;
            curr = code.next();
        }
        modified = checkExceptionsInIns(curr, handler, idx, excCons)
                   || modified;
        return modified;
    }

    static LocalVariable addLocalVar(Code code, String name, String type) {
        LocalVariableTable lTable = code.getLocalVariableTable(true);
        LocalVariable lv;
        lv = lTable.addLocalVariable(name, type);
        // work around bug in serp: set index manually
        int index = code.getNextLocalsIndex();
        lv.setLocal(index);
        code.setMaxLocals(index + 1);
        // difficult to set accurate start and end values
        lv.setStartPc(0);
        return lv;
    }

    void addExcBlockLocalVar(Code code) {
        excBlockLocalVar =
            CoverageTransformer.addLocalVar(code, "$excBlock", "I");
        code.beforeFirst();
        code.constant().setValue(-1);
        code.istore().setLocalVariable(excBlockLocalVar);
    }

    boolean transformHandlers(BCMethod method) {
        boolean modified = false;
        excBlockLocalVar = null;
        invocationTarget = null;
        dynCheckResult = null;
        instrSet = new TreeSet<InstrSetEntry>();
        handlerSet = new HashSet<Instruction>();
        Code code = method.getCode(false);
        ExceptionHandler[] handlers = code.getExceptionHandlers();
        for (int i = 0; i < handlers.length; i++) {
            if (handlers[i].getCatchType() != null) {
                // ignore null ("any") type handlers used for finally blocks
                Log.log("Analyzing handler for " +
                        handlers[i].getCatchType().getName() + " in " +
                        method.getDeclarer().getName() + "." +
                        method.getName(), Log.DEBUG);
                modified = checkExceptions(code, handlers, i) || modified;
            }
        }
        if (modified) {
            assert (excBlockLocalVar == null);
            addExcBlockLocalVar(code);
            // instrument all instructions in instrSet
            instrumentThrowableIns(code);
            instrumentHandlers(code);
            excBlockLocalVar.updateTargets();
            if (invocationTarget != null) {
                invocationTarget.updateTargets();
            }
            code.calculateMaxStack();
        }
        return modified;
    }

    /** Changes java.lang.Exception.printStackTrace() to
    (static) rt.enforcer.Reflect.printStackTrace(exc). */
    boolean transformPrintStackTrace(BCMethod method) {
        boolean modified = false;
        Code code = method.getCode(false);
        code.beforeFirst();
        while (code.hasNext()) {
            Instruction ins = code.next();
            if (ins.getOpcode() == 0xb6) { // 0xb6 = invokevirtual
                MethodInstruction mIns = (MethodInstruction) ins;
                BCClass cls = mIns.getMethodDeclarerBC();
                String methodName = mIns.getMethodName();
                if (methodName.equals("printStackTrace") &&
                        classIsException(cls)) {
                    code.remove();
                    mIns = code.invokestatic();
                    String[] mArgs = new String[] { "java.lang.Exception" };
                    mIns.setMethod("enforcer.rt.Reflect",
                                   "printStackTrace", "V", mArgs);
                }
            }
        }
        return modified;
    }

    static boolean classIsException(BCClass cls) {
        BCClass superClass;
        if (cls.getName().equals("java.lang.Exception")) {
            return true;
        } else {
            superClass = cls.getSuperclassBC();
            if (superClass != null) {
                return classIsException(superClass);
            }
            return false;
        }
    }

    /** Changes junit.framework.TestCase.init to
    rt.enforcer.ExceptionTestCase.init. */
    boolean transformTestCaseInit(BCMethod method) {
        boolean modified = false;
        Code code = method.getCode(false);
        code.beforeFirst();
        while (code.hasNext()) {
            Instruction ins = code.next();
            if (ins.getOpcode() == 0xb7) { // 0xb7 = invokespecial
                MethodInstruction mIns = (MethodInstruction) ins;
                String className = mIns.getMethodDeclarerName();
                String methodName = mIns.getMethodName();
                if (className.equals("junit.framework.TestCase") &&
                        methodName.equals("<init>")) {
                    mIns.setMethodDeclarer(NEW_TEST_SUPER_CLASS);
                    modified = true;
                }
            } else if (ins.getOpcode() == 0xbb) { // 0xbb = new
                ClassInstruction cIns = (ClassInstruction) ins;
                String className = cIns.getTypeName();
                if (className.equals("junit.framework.TestCase")) {
                    cIns.setType(NEW_TEST_SUPER_CLASS);
                    modified = true;
                }
            }
        }
        return modified;
    }

    boolean transformMethod(BCMethod method, boolean instrTries) {
        boolean modified = false;
        Code code = method.getCode(false);
        if (code == null) { // no code (e.g. abstract/native method)
            return false;
        }
        // assumption: suppressed method does not launch tests
        if (checkSuppressionList(method)) {
            Log.log("Skipping method " + method.getDeclarer().getName() +
                    "." + method.getName(), Log.DEBUG);
            return false;
        }
        currentMethod = method;
        modified = transformTestCaseInit(method) || modified;
        modified = transformPrintStackTrace(method) || modified;
        if (instrTries) {
            modified = transformHandlers(method) || modified;
        }
        modified = transformTestLaunch(code) || modified;
        return modified;
    }

    public byte[] transform(BCClass cls) {
        boolean modified = false;
        byte[] result;
        String superClassName;
        boolean instrTries = true;
        superClassName = cls.getSuperclassName();
        if ((superClassName.compareTo("junit.framework.TestCase")) == 0) {
            if (!(Options.getBool("instrTestCode"))) {
                Log.log("Will not instrument try blocks in test class " +
                        cls.getName() + "...", Log.DEBUG);
                instrTries = false;
            }
            // change super class to ExceptionTestCase to suppress
            // rethrown exceptions (false positives)
            Log.log("Changing super class of " + cls.getName(),
                    Log.DEBUG_LOWLEVEL);
            cls.setSuperclass("enforcer.rt.ExceptionTestCase");
            modified = true;
        }
        BCMethod[] methods = cls.getDeclaredMethods();
        for (int i = 0; i < methods.length; i++) {
            modified = transformMethod(methods[i], instrTries) || modified;
        }
        // Currently, assertions are not used in instrumented code.
        //if (modified) {
        //    enableAssertionCode (cls);
        //}
        if (modified) {
            result = cls.toByteArray();
            Log.log("New class file: " + result.length + " bytes.");
        } else {
            result = null;
        }
        return result;
    }

    /** Instrumentation front end; see <tt>java.lang.instrument</tt>. */
    public byte[] transform(String className, byte[] classfileBuffer) {

        byte[] result;
        if (className.startsWith("java/") ||
                className.startsWith("sun/") ||
                className.startsWith("enforcer/rt/") ||
                className.startsWith("junit/")) {
            result = null;
        } else {
            // cannot use getSuperClass() because super class has
            // not yet been loaded
            Log.log("Transforming " + className +
                    " (" + classfileBuffer.length + " bytes)...");
            Project p = new Project();
            try {
                ByteArrayInputStream input =
                    new ByteArrayInputStream(classfileBuffer);
                result = transform(new Project().loadClass(input));
            } catch (Exception e) {
                e.printStackTrace();
                result = null;
            }
        }
        if ((result != null) && (Options.get("dumpPath") != null)) {
            dumpResult(className, result);
        }

        return result;
    }

    /* Checks whether field $assertionsDisabled and code in
    <tt>&lt;clinit&gt;</tt> exists to use assertions. */
    /* currently not needed, but previously tested */
    /*
    void enableAssertionCode(BCClass cls) {
    if (cls.getDeclaredField("$assertionsDisabled") != null) {
     return;
    }
    BCField adField = cls.declareField("$assertionsDisabled", "Z");
    adField.setFinal(true);
    adField.setStatic(true);

    // create static initializer if necessary
    BCMethod staticInitializer = cls.getDeclaredMethod("<clinit>");
    String[] emptyArgs = new String[0];
    if (staticInitializer == null) {
     staticInitializer =
    cls.declareMethod("<clinit>", "V", emptyArgs);
     staticInitializer.setStatic(true);
     staticInitializer.getCode(true).vreturn();
    }
    Code code = staticInitializer.getCode(false);

    // assertion init. code
    code.constant().setValue(cls);
    MethodInstruction ins2 = code.invokevirtual();
    ins2.setMethod("java.lang.Class", "desiredAssertionStatus",
         "Z", emptyArgs);
    IfInstruction branch1 = code.ifne();
    code.constant().setValue(1);
    JumpInstruction go2 = code.go2();
    Instruction ifTarget = code.constant().setValue(0);
    Instruction go2Target = 
     code.putstatic().setField(cls.getName(),
          "$assertionsDisabled", "Z");

    branch1.setTarget(ifTarget);
    go2.setTarget(go2Target);
    if (code.getMaxStack() < 1) {
     code.setMaxStack(1);
    }
    }
    */

    /** Adds an entry to <tt>instrSet</tt>. Note: the caller has
        to ensure that only one entry per instruction is added. */
    void addToInstrSet(Instruction ins, Class excType,
                       Constructor excCons, int rank) {
        InstrSetEntry entry = new InstrSetEntry();
        entry.ins = ins;
        entry.excType = excType;
        entry.excCons = excCons;
        entry.rank = rank;
        assert (!instrSet.contains(entry));
        instrSet.add(entry);
    }

    public void dumpResult(String className, byte[] result) {
        try {
            String dirName =
                new String(Options.getString("dumpPath") + "/" + className);
            dirName = dirName.substring(0, dirName.lastIndexOf("/"));
            File dir = new File(dirName);
            dir.mkdirs();
            FileOutputStream f =
                new FileOutputStream(Options.getString("dumpPath") + "/" +
                                     className + ".class");
            f.write(result);
            f.close();
        } catch (java.io.IOException e) {
            Log.errLog("Could not open dump file: " + e);
        }
    }
}
