package enforcer;

/* $Id: Main.java 579 2007-05-30 03:02:24Z cartho $ */

import enforcer.etc.Options;
import enforcer.instr.CoverageTransformer;
import enforcer.log.Log;
import java.io.File;
import java.util.Iterator;
import java.util.TreeSet;
import serp.bytecode.BCClass;
import serp.bytecode.Project;

public class Main {
    public final static int OPT_DIR = 1;
    public final static int OPT_INSTR_TEST = 2;
    public final static int OPT_NO_INSTR_TEST = 3;
    public final static int OPT_LOGLEVEL = 4;
    public final static int OPT_USAGE = 5;
    public final static int OPT_SHOW = 6;

    public static boolean printUsage = false;
    public static boolean showOpts = false;

    static CoverageTransformer ct = new CoverageTransformer();

    /** Class implementing <tt>main</tt> method for off-line
	instrumentation. Use for backwards compatibility with JDK
	1.4 and when bugs in JDK 1.5 prevent successful instrumentation.
	*/
    public static final void main(String[] args) {
	int i;
	i = 0;
	try {
	    while ((i < args.length) &&
		   (args[i].charAt(0) == '-')) {
		i = parseOption(args, i);
		i++;
	    }
	    // check for crucial options
	    // dumpPath
	    checkValues();
	} catch (IllegalArgumentException e) {
	    System.err.println("Error: " + e.getMessage() + ".");
	    System.out.println();
	    printUsage = true;
	}
	if (printUsage) {
	    usage();
	}
	if (showOpts) {
	    if (printUsage) {
		System.out.println();
	    }
	    showValues();
	}
	if (printUsage) {
	    return;
	}
	useArgs(args, i);
    }

    public static void checkValues() throws IllegalArgumentException {
	if (Options.get("dumpPath") == null) {
	    throw new IllegalArgumentException("Output directory " +
					       "(-d <outdir>) not given");
	}
    }

    public static void showValues() {
	Iterator it = new TreeSet<String>(Options.keySet()).iterator();
	System.out.println("Key                  value");
	System.out.println("-----------------------------");
	String spc = new String("                    ");
	while (it.hasNext()) {
	    String key = (String) it.next();
	    int len = key.length();
	    System.out.print(key + spc.substring(len) + " ");
	    Object data = Options.get(key);
	    Class cls = (data != null) ? data.getClass() : null;
	    if ((cls == Boolean.class) || (cls == Integer.class) ||
		(cls == String.class)) {
		System.out.println(data);
	    } else {
		if (data == null) {
		    System.out.println("null");
		} else {
		    System.out.println("non-null");
		}
	    }
	}
    }

    public static int parseOption(String[] args, int pos)
	throws IllegalArgumentException {
	String option;
	String value = "";
	int idx;
	int opt;
	boolean valSet = false;

	option = args[pos].charAt(1) == '-' ?
	    args[pos].substring(2) : args[pos].substring(1);
	if ((idx = option.indexOf('=')) != -1) {
	    value = option.substring(idx + 1);
	    option = option.substring(0, idx);
	    valSet = true;
	}
	opt = checkOption(option);
	boolean needsVal = valueRequired(opt);
	if (!needsVal && valSet) {
	    throw new IllegalArgumentException("No values for option \"" +
					       option + "\" supported");
	}
	if (needsVal && !valSet) {
	    if (args.length < pos + 2) {
		throw new IllegalArgumentException("Missing argument " +
						   "for option \"" + option +
						   "\"");
	    }
	    value = args[++pos];
	}
	evalOption(opt, value);
	return pos;
    }

    public static void evalOption(int optId, String value)
	throws IllegalArgumentException {
	switch (optId) {
	    case OPT_DIR:
		Options.set("dumpPath", value);
		break;
	    case OPT_LOGLEVEL:
		int logLevel = (new Integer(value)).intValue();
		if ((logLevel < 0) || (logLevel > 4)) {
		    throw new IllegalArgumentException("Log level must be " + 
						       "between 0 and 4");
		}
		Options.set("logLevel", logLevel);
		break;
	    case OPT_NO_INSTR_TEST:
		assert (value.compareTo("") == 0);
		Options.set("instrTestCode", false);
		break;
	    case OPT_INSTR_TEST:
		assert (value.compareTo("") == 0);
		Options.set("instrTestCode", true);
		break;
	    case OPT_USAGE:
		assert (value.compareTo("") == 0);
		printUsage = true;
		break;
	    default:
		assert (value.compareTo("") == 0);
		assert (optId == OPT_SHOW);
		showOpts = true;
	}
    }

    public static int checkOption(String option)
	throws IllegalArgumentException {
	if ((option.compareTo("d") == 0) ||
	    (option.compareTo("dir") == 0)) {
	    return OPT_DIR;
	}
	if ((option.compareTo("h") == 0) ||
	    (option.compareTo("help") == 0)) {
	    return OPT_USAGE;
	}
	if ((option.compareTo("s") == 0) ||
	    (option.compareTo("show") == 0)) {
	    return OPT_SHOW;
	}
	if (option.compareTo("instr-test") == 0) {
	    return OPT_INSTR_TEST;
	}
	if (option.compareTo("no-instr-test") == 0) {
	    return OPT_NO_INSTR_TEST;
	}
	if (option.compareTo("log-level") == 0) {
	    return OPT_LOGLEVEL;
	}
	throw new IllegalArgumentException("Option \"" + option + "\" not " +
					   "supported.");
    }

    public static boolean valueRequired(int optId) {
	switch(optId) {
	    case OPT_DIR:
	    case OPT_LOGLEVEL:
		return true;
	    default:
		assert ((optId == OPT_INSTR_TEST) ||
			(optId == OPT_NO_INSTR_TEST) ||
			(optId == OPT_USAGE) ||
			(optId == OPT_SHOW));
		return false;
	}
    }

    public static void usage() {
	System.out.print("Usage: ");
	System.out.print("java -jar enforcer.jar ");
	System.out.println("[-d <outdir>] [--options...] class [class...]");
	System.out.println();
	System.out.println("where options include:");
	System.out.println("-h or --help      show this screen");
	System.out.println("-s or --show      show configuration");
	System.out.println("-d <outdir>       to set output directory");
	System.out.println("                  (mandatory if " +
			   "ENFORCER_INSTR_DUMP not set)");
	System.out.println("--instr-test      instrument test code as well");
	System.out.println("--no-instr-test   do not instrument " +
			   "test code as well");
	System.out.println("--log-level <l>   set log level (0 .. 4)");
    }

    public static void useArgs(String[] args, int pos) {
	while (pos < args.length) {
	    transform(args[pos]);
	    pos++;
	}
	Log.log(CoverageTransformer.excBlock +
		" try/catch blocks instrumented.", Log.INFO);
    }

    public static void transform(String className) {
	BCClass cls = new Project().loadClass(new File(className));
	byte[] result = ct.transform(cls);
	if (result != null) {
	    String noClassStr = className.substring(0, className.length() - 6);
	    ct.dumpResult(noClassStr, result);
	}
    }
}
